/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Mouse Gestures for Mozilla.
 *
 * The Initial Developer of the Original Code is Jens Bannmann.
 * Portions created by the Initial Developer are Copyright (C) 2003
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s) (alphabetical order):
 *  Jens Bannmann <jens.b@web.de>
 *  Jochen <bugs@krickelkrackel.de>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

var mgStorage = {
  RDF : Components.classes["@mozilla.org/rdf/rdf-service;1"]
        .getService(Components.interfaces.nsIRDFService),

  RDFCUtils : Components.classes["@mozilla.org/rdf/container-utils;1"]
              .getService()
              .QueryInterface(Components.interfaces.nsIRDFContainerUtils),

  NS : "http://optimoz.mozdev.org/gestures-rdf#",

  userDS : null,
  defaultMappingsDS : null,

  getUserRDF : function() {
    if (this.userDS) {
      return this.userDS;
    } else {
      var mgUserFile = this.getProfileDir();
      mgUserFile.append("mousegestures.rdf");
      this.userDS = this.RDF.GetDataSourceBlocking(this.fileToURL(mgUserFile).spec);
      return this.userDS;
    }
  },

  getDefaultMappings : function() {
    if (!this.defaultMappingsDS) {
      this.defaultMappingsDS = this.RDF
          .GetDataSourceBlocking("chrome://mozgest/content/defaultMappings.rdf");
    }
    return this.defaultMappingsDS;
  },

  initUserRDF : function() {
    var dsUser = this.getUserRDF();

    var usInfo = this.RDF.GetResource("urn:mozgest:userstore");
    // determine which mozgest-version saved this userstore. default to 0 if
    //    userstore doesnt exist - this will trigger installation of mappings below
    var userstoreVersion = this.getProperty(dsUser, usInfo, "creatorVersion") || 0;

    var dsDefault = this.getDefaultMappings();

    // get release info from property file
    var releaseVersion = mgGetBuildProperties().GetStringFromName("mozgest.version");

    // ensure both versions are strings, so the comparison works correctly
    releaseVersion="" + releaseVersion;
    userstoreVersion="" + userstoreVersion;

    if (releaseVersion > userstoreVersion) {
      this.unassertLiteral(dsUser, usInfo, "creatorVersion", userstoreVersion);
      this.assertLiteral(dsUser, usInfo, "creatorVersion", releaseVersion);
      // enumerate all resources from defaultMappings file
      var elements = dsDefault.GetAllResources();
      while (elements.hasMoreElements()) {
        var el = elements.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
        // only process Seqs
        if(this.RDFCUtils.IsSeq(dsDefault, el)) {
          var sourceSeq = this.newContainer();
          sourceSeq.Init(dsDefault, el);
          var targetSeq = this.newContainer();
          // init current seq in userstore, or create it if it doesn't exist
          var emptySeq = false;
          try {
            targetSeq.Init(dsUser, el);
          } catch (e) {
            this.RDFCUtils.MakeSeq(dsUser, el);
            targetSeq.Init(dsUser, el);
            emptySeq = true;
          }
          var mappingsEnum = sourceSeq.GetElements();
          while (mappingsEnum.hasMoreElements()) {
            // get mapping data
            var mapping = mappingsEnum.getNext();

            // determine which mozgest version introduced this mapping.
            //    default to 0.4 if no "since" property present
            var mSince = this.getProperty(dsDefault, mapping, "since") || 0.4;

            // if mapping is newer than userstoreVersion, install it
            if (mSince > userstoreVersion) {
              if (emptySeq ||
                  !this.findMapping(this.getProperty(dsDefault, mapping, "code"),
                                    targetSeq, dsUser)) {
                this.installMapping(mapping, targetSeq);
              }
            }
          }
        }
      }
      dsUser.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
    }
    dsUser.RemoveObserver(mgMappingsObserver);
    dsUser.AddObserver(mgMappingsObserver);
  },

  // lookup mapping by code
  findMapping : function(code, targetContainer, datasource) {
    var mappingsEnum = targetContainer.GetElements();
    while (mappingsEnum.hasMoreElements()) {
      var mapping = mappingsEnum.getNext();
      if (this.getProperty(datasource, mapping, "code") == code) {
        return mapping;
      }
    }
    return false;
  },

  // installs a mapping from defaultMappings to userstore.
  installMapping : function(mapping, targetContainer) {
    var dsUser = this.getUserRDF();
    var dsDefault = this.getDefaultMappings();

    var mCode = this.getProperty(dsDefault, mapping, "code");
    var mFunction = this.getProperty(dsDefault, mapping, "function");
    var mAppearance = this.getProperty(dsDefault, mapping, "appearance");

    node = this.RDF.GetAnonymousResource();
    this.saveMapping(dsUser, targetContainer, node,
                     mCode, mAppearance, mFunction, "", "", "");
  },

  // shared function for saving mappings
  saveMapping : function(aDataSource, aContainer, aNode,
                         aCode, aAppearance, aFunction,
                         aCustom, aBMOpen, aName) {
    aContainer.AppendElement(aNode, true);
    this.assertLiteral(aDataSource, aNode, "code", aCode);
    if (aAppearance) {
      this.assertLiteral(aDataSource, aNode, "appearance", aAppearance);
    }
    if (aFunction) {
      this.assertLiteral(aDataSource, aNode, "function", aFunction);
    }
    if (aCustom) {
      this.assertLiteral(aDataSource, aNode, "custom", aCustom);
    }
    if (aBMOpen) {
      this.assertLiteral(aDataSource, aNode, "bmOpenIn", aBMOpen);
    }
    if (aName) {
      this.assertLiteral(aDataSource, aNode, "name", aName);
    }
    if ("mindAdded" in aDataSource) {
      aDataSource.mindAdded(aNode);
    }
  },

  // Deletes a node from the userstore
  deleteNode : function(aNode, aDataSource) {
    if (!aNode) {
      return;
    }
    // WARNING: this is intended only for nodes inside a container. behaviour
    // with nodes that have multiple or non-container inward-arcs is untested!
    this.arcs = aDataSource.ArcLabelsIn(aNode);
    while(this.arcs.hasMoreElements()) {
      this.arc = this.arcs.getNext();
      var parent = aDataSource.GetSource(this.arc, aNode, true);
      if (mgStorage.RDFCUtils.IsContainer(aDataSource, parent)) {
        var container = mgStorage.newContainer();
        container.Init(aDataSource, parent);
        container.RemoveElement(aNode, true);
      }
    }
    this.arcs = aDataSource.ArcLabelsOut(aNode);
    while(this.arcs.hasMoreElements()) {
      this.arc = this.arcs.getNext();
      var targets = aDataSource.GetTargets(aNode, this.arc, true);
      while (targets.hasMoreElements()) {
        var target = targets.getNext();
        aDataSource.Unassert(aNode, this.arc, target, true);
      }
    }
    if ("mindAdded" in aDataSource) {
      aDataSource.mindDeleted(aNode);
    }
  },

  processURL : function (url, docLoc) {
    url = url.substr(10);
    var parts = url.split("/");
    // check whether command is valid
    switch (parts[0]) {
      case "addmapping":                            //check whitelist for import not ready yet
        this.importMapping(parts);
        break;
      case "patch":
        if (this.checkWhiteList(parts[0], docLoc))  //check whitelist for patches before
          this.importMapping(parts);                //calling importMapping(). The whitelist
        break;                                      //for patches is and will stay hardcoded
      default:
        var msg = mgBundle.getString("unknownCommand");
        msg = msg.replace(/%COMMAND%/, parts[0]);
        alert(msg);
    }
  },

  checkWhiteList : function (module, docLoc) {
     if (module == "patch") {
       if (docLoc.host == "mousegestures.org" ||
           docLoc.host == "www.mousegestures.org" ||
           docLoc.host == "dev.mousegestures.org" ||
           docLoc.host == "bugzilla.mozdev.org" ||
           docLoc.protocol == "file:")
         return true;
       var msg = mgBundle.getString("noPermissionToImport");
       msg = msg.replace(/%MESSAGE%/, docLoc.host);
       alert(msg);
       return false;
     }
     //if (module == "addmapping")
     return true;
  },

  importMapping : function (parts) {
    //parts[0] can be "patch" or "addmapping"
    //"addmapping"-whitelist needs UI!!!
    var module = "Import";

    // check whether the given windowtype is known
    var prefix = this.findPrefix(parts[1]);
    if (!prefix) {
      var msg = mgBundle.getString("unknownWindowType");
      msg = msg.replace(/%WINDOWTYPE%/, parts[1]);
      alert(msg);
      return;
    }
    var validCode = false;
    var code = "";
    var appearance;
    var gestureRegExp = /^(\*{0,1}[1379DLUR]+){1}(;\*{0,1}[1379DLUR]+){0,1}$/;
    if (parts[2].match(/^:[0-2][0-2\+\-]$/)) {
      validCode = true;
      code = parts[2];
    } else if (gestureRegExp.exec(parts[2])) {
      validCode = true;
      code = RegExp.$1;
      appearance = RegExp.$2.substr(1);
      if (this.compactCode(code) != code
          || (appearance && this.compactCode(appearance) != code) ) {
        validCode = false;
      }
    }

    if (parts[0] == "patch") {
      code = "UUUUUU"; // brute force
      validCode = true;
    }

    if (!validCode) {
      alert(mgBundle.getString("invalidCode"));
      return;
    }

    var func = "";
    var name = "";
    var custom = "";
    if (parts[3] == "custom") {
      name = decodeURIComponent(parts[4]);
      custom = decodeURIComponent(parts[5]);
    } else {
      // checks whether function is known and supports this window type
      var c = this.functionSupportsType(parts[3], parts[1], prefix);
        if(c == -1) {
        this.msg = mgBundle.getString("unknownFunction");
        this.msg = this.msg.replace(/%FUNCTION%/, parts[3]);
        alert(this.msg);
        return;
      } else if (c == 0) {
        this.msg = mgBundle.getString("functionIncompatible");
        this.msg = this.msg.replace(/%FUNCTION%/, parts[3]);
        this.msg = this.msg.replace(/%WINDOWTYPE%/, parts[1]);
        alert(this.msg);
        return;
      }
      func = parts[3];
    }
    code = mgMappingLocalizer.localize(code);
    appearance = mgMappingLocalizer.localize(appearance);
    window.openDialog("chrome://mozgest/content/pref/edit-mapping.xul",
                      "mozgest:editmapping", "chrome,centerscreen,modal=yes",
                      module, "", code, appearance, func, name, custom, "",
                      prefix, parts[1]);
  },

  receiveEditedMapping : function (uri, code, appearance,
                                   func, name, custom, bmOpenIn, aWindowType) {
    var ds = this.getUserRDF();
    var stdCode = mgMappingLocalizer.deLocalize(code);
    var stdApp = mgMappingLocalizer.deLocalize(appearance);

    // prepare container for this windowtype
    var mappingsContainer = this.newContainer();
    var containerNode = this.RDF.GetResource("urn:mozgest:mappings:"
                                             + aWindowType);
    try {
      mappingsContainer.Init(ds, containerNode);
    } catch (e) {
      // create mappings container in user store
      this.RDFCUtils.MakeSeq(ds, containerNode);
      this.newContainer().Init(ds, containerNode);
    }
    if (!this.duplicateCheck(stdCode, mappingsContainer, uri, ds)) {
      return false;
    }
    if (uri != "") {
      mgStorage.deleteNode(mgStorage.RDF.GetResource(uri), ds);
    }

    node = this.RDF.GetAnonymousResource();
    this.saveMapping(ds, mappingsContainer, node,
                     stdCode, stdApp, func,
                     encodeURI(custom), bmOpenIn, name);
    return true;
  },

  compactCode : function (aCode) {
    var compacted = aCode.substr(0, 1);
    for(var i = 1; i < aCode.length; i++) {
      var c = aCode.substr(i, 1);
      if (c != compacted.substr(compacted.length - 1, 1)) {
        compacted += c;
      }
    }
    return compacted;
  },

  findPrefix : function (aType) {
    var dsTypes = mgStorage.RDF.GetDataSourceBlocking("chrome://mozgest/content/windowTypes.rdf");
    typesList = Components.classes['@mozilla.org/rdf/container;1']
                       .createInstance()
                       .QueryInterface(Components.interfaces.nsIRDFContainer);
    try {
      typesList.Init(dsTypes, mgStorage.RDF.GetResource("urn:mozgest:windows:list"));
    } catch (e) {
      mgCommon.dump("MozGest: Could not init typesList. " + e + "\n");
      return false;
    }
    var typesEnum = typesList.GetElements();
    while (typesEnum.hasMoreElements()) {
      var type = typesEnum.getNext()
                 .QueryInterface(Components.interfaces.nsIRDFResource);
      if (!mgStorage.getProperty(dsTypes, type, "separator")) {
        var typeCode = mgStorage.getProperty(dsTypes, type, "wtype");
        if (typeCode == aType) {
          return mgStorage.getProperty(dsTypes, type, "prefix");
        }
      }
    }
    return false;
  },

  // returns: -1 for not found, 0 for unsupported windowtype, 1 on success
  functionSupportsType : function (aFuncName, aWindowType, aWindowTypePrefix) {
    var ret = this.functionKnown(aFuncName);
    if (ret) {
      var fSupports = ","+ret+",";
      if (aFuncName.indexOf("mg" + aWindowTypePrefix + "_") == 0
          || aFuncName.indexOf("mgW_") == 0
          || fSupports.indexOf("," + aWindowType + ",") != -1) {
        return 1;
      } else {
        return 0;
      }
    }
    return -1;
  },

  // returns: false for unknown, "default" or comma-separated list of types if
  //          function is known
  functionKnown : function (aFuncName) {
    var dsFunc = mgStorage.RDF.GetDataSourceBlocking("chrome://mozgest/content/functions.rdf");
    var functionList = Components.classes['@mozilla.org/rdf/container;1']
                             .createInstance()
                             .QueryInterface(Components.interfaces.nsIRDFContainer);
    try {
      functionList.Init(dsFunc, mgStorage.RDF.GetResource("urn:mozgest:functions"));
    } catch (e) {
      mgCommon.dump("MozGest: Could not init functionList. " + e + "\n");
      return false;
    }
    var funcEnum = functionList.GetElements();
    while (funcEnum.hasMoreElements()) {
      var func = funcEnum.getNext()
                 .QueryInterface(Components.interfaces.nsIRDFResource);
      var fName = mgStorage.getProperty(dsFunc, func, "func");
      if (fName == aFuncName) {
        var supports = mgStorage.getProperty(dsFunc, func, "supports");
        return supports ? supports : "default";
      }
    }
    return false;
  },

  duplicateCheck : function(aCode, aTargetContainer, aURI, aDataSource) {
    //aCode will always be "UUUUUU" for a patch
    if (aCode == "UUUUUU")
      return true;

    var res = this.findMapping(aCode, aTargetContainer, aDataSource);
    if (res && aURI != res.Value) {
      var msg = mgBundle.getString("overwriteWarning");
      var name = this.getProperty(aDataSource, res, "name");
      if (!name) {
        name = mgBundle.getString(this.getProperty(aDataSource, res, "function"));
      }
      msg = msg.replace(/%MAPPINGNAME%/,
                        name);
      var dc = aDataSource.mgLocalized ? aCode
                                       : mgMappingLocalizer.localize(aCode);
      msg = msg.replace(/%CODE%/, dc);
      if (!confirm(msg)) {
        return false;
      } else {
        this.deleteNode(res, aDataSource);
      }
    }
    return true;
  },

  readRDFGestureTable : function(component, readDefault) {
    // readDefault is used by the print window
    // get our datasource
    var dsource;
    if (!readDefault)
      dsource = this.getUserRDF();
    else
      dsource = this.getDefaultMappings();

    // instantiate a container object and its root node
    var mappingsContainer = this.newContainer();
    var rootnode = this.RDF.GetResource("urn:mozgest:mappings:" + component);

    try {
      mappingsContainer.Init(dsource, rootnode);
    } catch (e) {
      // no existing mappings
      return;
    }

    var prefix = mgGetWindowTypePrefix(component);
    var nodesToDelete = new Array();
    var keyWordsToConvert = new Array();
    var patchesToApply = new Array();

    var enumerator = mappingsContainer.GetElements();
    while (enumerator.hasMoreElements()) {
      var mapping = enumerator.getNext()
                    .QueryInterface(Components.interfaces.nsIRDFResource);
      var entry = new Mapping();
      entry.resource = mapping.Value;
      entry.code = this.getProperty(dsource, mapping, "code");
      if (entry.code == "UUUUUU") {
        entry.code = entry.code + patchesToApply.length;
        entry.isPatch = true;
      }
      entry.appearance = this.getProperty(dsource, mapping, "appearance");
      entry.func = this.getProperty(dsource, mapping, "function");
      entry.custom = decodeURI(this.getProperty(dsource, mapping, "custom"));
      entry.bmOpenIn = this.getProperty(dsource, mapping, "bmOpenIn");
      entry.component = component;
      entry.countBase = Math.abs(this.getProperty(dsource, mapping, "count")) || 0;
      if (entry.func == null
          || this.functionSupportsType(entry.func, component, prefix) == 1) {
        entry.name = (entry.func) ? mgBundle.getString(entry.func)
                                  : this.getProperty(dsource, mapping, "name");
        addGesture(entry);
      } else {
        // function no longer supported, schedule mapping for deletion
        mgCommon.dump("readRDFGestureTable: " + entry.func + " not supported on '"
             + component + "' (" + prefix + "), mapping deleted\n");
        nodesToDelete.push(mapping);
      }

      if (mgWindowType) {
        if (entry.custom.indexOf("mozgest_iKey:") == 0)
          keyWordsToConvert.push(entry);

        if (entry.isPatch) {
          try {
            var rVersion = mgGetBuildProperties().GetStringFromName("mozgest.version")
            var endpatch = entry.custom.indexOf("//mozGestPatch####")
            var pVersion = entry.custom.substring(21, endpatch - 1)

            if (pVersion == rVersion)
              patchesToApply.push(entry);
            else
              nodesToDelete.push(mapping);
          }
          catch (e) {
            nodesToDelete.push(mapping)
          }
        }
      }

      if (nodesToDelete.length > 0) {
        // Disable the observer to prevent multiple loadGestureTable calls
        // while we delete mappings to no longer supported functions
        mgMappingsObserver.enabled = false;

        // If we get here, we have at least one node to delete
        mapping = nodesToDelete.pop(); // Get the first node
        do {
          this.deleteNode(mapping, dsource);
          mapping = nodesToDelete.pop(); // Get the next node from the list
        }   while (mapping != null);       // Continue until all nodes are deleted

        mgMappingsObserver.enabled = true;
      }

      if (keyWordsToConvert.length > 0 && mgBMService.enabled)
         setTimeout(this.convertKeyWords, 500,
                    keyWordsToConvert, dsource, mappingsContainer);

      if (patchesToApply.length > 0)
         setTimeout(this.applyPatches, 1500, patchesToApply);
    }
  },

  applyPatches : function (patches) {
    for (i = 0; i < patches.length; i++) {
      try {
        eval(patches[i].custom)
      }
      catch (e) {}
    }
  },

  convertKeyWords : function (kw, dsource, mappingsContainer) {
    mgMappingsObserver.enabled = false;

    for (i = 0; i < kw.length; i++) {
      var keyWord = kw[i].custom.substring(13, kw[i].custom.length).toLowerCase();
      var newRes = mgBMService.Book.GetSource(mgBMService.KeyW,
                               mgBMService.RDFS.GetLiteral(keyWord), true);

      if (!newRes)
        newRes = "rdf:#mozgest"; //keep mapping, but convert it to an unknown bookmark
      else
        newRes = newRes.QueryInterface(Components.interfaces.nsIRDFResource).Value;

      mgStorage.deleteNode(mgStorage.RDF.GetResource(kw[i].resource), dsource);
      var kwNode = mgStorage.RDF.GetAnonymousResource();

      mgStorage.saveMapping(dsource, mappingsContainer, kwNode,
                            kw[i].code, kw[i].appearance, kw[i].func,
                            newRes, "0", kw[i].name);
    }

    mgMappingsObserver.enabled = true;
    mgMappingsObserver.delayedUpdate();
  },

  shutdown : function() {
    store = window.mgStorage;
    mgCommon.dump("MozGest: Saving gesture counts\n");
    var dsource = store.getUserRDF();
    for (var code in gesturesTable) {
      mapping = gesturesTable[code];
      if (mapping.countCur != 0) {
        // get currently saved value
        var node = store.RDF.GetResource(mapping.resource);
        var old = Math.abs(store.getProperty(dsource, node, "count")) || 0;

        // add current session count and save it
        var n = old + mapping.countCur;
        store.changeLiteral(dsource, node, "count", old, n);
      }
    }
    mgStorage.userDS.RemoveObserver(mgMappingsObserver);
  },


  /*** convenience functions for RDF access ***/

  newContainer : function() {
    return Components.classes['@mozilla.org/rdf/container;1'].createInstance()
           .QueryInterface(Components.interfaces.nsIRDFContainer);
  },

  assertLiteral : function(dsource, node, propName, value) {
    dsource.Assert(node, this.RDF.GetResource(this.NS + propName),
                   this.RDF.GetLiteral(value), true);
  },

  unassertLiteral : function(dsource, node, propName, value) {
    dsource.Unassert(node, this.RDF.GetResource(this.NS + propName),
                   this.RDF.GetLiteral(value), true);
  },

  changeLiteral : function(dsource, node, propName, oVal, nVal) {
    dsource.Change(node, this.RDF.GetResource(this.NS + propName),
                   this.RDF.GetLiteral(oVal), this.RDF.GetLiteral(nVal));
  },

  getProperty : function (dsource, obj, propName) {
    // get nsIRDFNode
    var literal = dsource.GetTarget(obj,
                                    this.RDF.GetResource(this.NS + propName),
                                    true);
    if (literal) {
      // convert to nsIRDFLiteral and get the value
      return literal.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
    } else {
      return null;
    }
  },


  /*** profile & URL handling ***/

  getProfileDir : function() {
    // get a File representing user's profile dir
    var profileInternal = Components.classes['@mozilla.org/file/directory_service;1']
                                    .getService(Components.interfaces.nsIProperties);
    return profileInternal.get('ProfD', Components.interfaces.nsILocalFile);
  },


  fileToURL : function(aFile) {
    var fileHandler = Components.classes["@mozilla.org/network/io-service;1"]
                        .getService(Components.interfaces.nsIIOService);
    return fileHandler.newFileURI(aFile);
  }

}



/*** observer stuff ***/

var mgMappingsObserver = new Object();
mgMappingsObserver = {
  updateTimer : null,
  enabled : true,

  // throughout this observer, we need the prefix "window." to access
  // mozgest's global variables - it seems we're somehow "outside" of the window

  delayedUpdate : function() {
    window.clearTimeout(this.updateTimer);
    this.updateTimer = window.setTimeout(loadGestureTable, 500, true);
  },

  onAssert: function(aDataSource, aSource, aProperty, aTarget) {
    if (aProperty.Value.indexOf(window.mgStorage.NS) == 0 && this.enabled) {
      this.delayedUpdate();
    }
  },

  onUnassert : function(aDataSource, aSource, aProperty, aTarget) {
    if (aProperty.Value.indexOf(window.mgStorage.NS) == 0 && this.enabled) {
      this.delayedUpdate();
    }
  },

  onChange: function(aDataSource, aSource, aProperty, aTarget, aOldTarget,
                     aNewTarget) {
    if (aProperty.Value && aProperty.Value.indexOf(window.mgStorage.NS) == 0
        && aProperty.Value != window.mgStorage.NS + "count"
        && this.enabled) {
      this.delayedUpdate();
    }
  },

  onMove: function(aDataSource, aOldSource, aNewSource, aProperty,
  aTarget) {
  },

  beginUpdateBatch: function(aDataSource) {
  },

  endUpdateBatch: function(aDataSource) {
  }

}