/* Provides a minimal common API for WHATWG's canvas element and wz_jsgraphics.
 * Automatically loads the latter as needed, by inserting a script element.
 *
 * Created by Jens Bannmann <jens.b@web.de>, this file is in public domain.
 *
 * 'graphics' object:
 * .createCanvasElement(id, w, h)    Creates element for insertion into DOM
 * .getContext(id)                   Gets drawing context for canvas
 *
 * Drawing context:
 * .setColor(html colorstring)       .setColor("#ff0000"); or .setColor("red");
 * .setStroke(linewidth)             .setStroke(2);    guaranteed maximum is 3px
 * .drawLine(x1, y1, x2, y2)         .drawLine(10,20,110,120);
 * .fillCircle(centerX, centerY, d)  .fillCircle(10, 10, 5);   d=diameter
 * .fillRect(x, y, width, height)    .fillRect(30, 70, 40, 10);
 * .paint()                          renders the canvas (implementations might
 *                                   render beforehand, but you must call this)
 */
var wzGraphics =
{
  createCanvasElement: function(id, width, height) {
    var el = document.createElement("div");
    el.setAttribute("id", id);
    el.style.position = "relative";
    el.style.width = width + "px";
    el.style.height = height + "px";
    return el;
  },
  getContext: function(canvasID) {
    return new wzContextWrapper(canvasID);
  }
}

function wzContextWrapper(canvasID)
{
  this.ctx = new jsGraphics(canvasID);
  this.drawLine = function(x1, y1, x2, y2) { this.ctx.drawLine(x1, y1, x2, y2); }
  this.fillRect = function(x, y, w, h) { this.ctx.fillRect(x, y, w, h); }
  this.paint = function() { this.ctx.paint(); }
  this.setStroke = function(width) { this.ctx.setStroke(width); }
  this.setColor = function(color) { this.ctx.setColor(color); }
  this.beginPath = function() {}

  this.fillCircle = function(centerX, centerY, diameter)
  {
                  this.ctx.fillOval(centerX - Math.floor(diameter / 2), centerY - Math.floor(diameter / 2), diameter, diameter);
  }

  if ("setPrintable" in this.ctx) this.ctx.setPrintable(true);
}

var nativeGraphics =
{
  createCanvasElement: function(id, width, height) {
    var el = document.createElement("canvas");
    el.setAttribute("id", id);
    el.setAttribute("width", width);
    el.setAttribute("height", height);
    return el;
  },
  getContext: function(canvasID) {
    return new nativeContextWrapper(canvasID);
  }
}

function nativeContextWrapper(canvasID)
{
  this.ctx = document.getElementById(canvasID).getContext("2d");
  this.drawLine = function(x1, y1, x2, y2)
  {
    this.ctx.moveTo(x1 + 0.5, y1 + 0.5);
    this.ctx.lineTo(x2 + 0.5, y2 + 0.5);
    this.ctx.stroke();
  };
  this.fillCircle = function(x, y, diameter)
  {
    this.ctx.arc(x + 0.5, y + 0.5, (diameter / 2), 0.1, 3, false);
    this.ctx.arc(x + 0.5, y + 0.5, (diameter / 2), 3, 0.1, false);
    this.ctx.fill();
  };
  this.fillRect = function(x, y, w, h)
  {
    this.ctx.lineJoin = "miter";
    this.ctx.fillRect(x, y, w, h);
  };
  this.paint = function()
  {
  };
  this.setStroke = function(width)
  {
    this.ctx.lineWidth = width;
  };
  this.setColor = function(color)
  {
    this.ctx.strokeStyle = color;
    this.ctx.fillStyle = color;
  };
  this.beginPath = function()
  {
    this.ctx.beginPath();
  };
}

var graphics;
if ("getContext" in document.createElement("canvas"))
{
  graphics = nativeGraphics;
}
else
{
  graphics = wzGraphics;
  document.write('<script type="text/javascript" src="wz_jsgraphics.js"></script>');
}