/*
 * chrome://tabmixplus/content/session/sessionStore.js
 *
 * original code by onemen
 *
 */

var TMP_SessionStore = {
   /**
    * @brief       - Add attribute to nsSessionStore persistTabAttribute.
    *
    * @param doInit   a Boolean value - true when we need to init nsISessionStore.
    *
    * @returns        Nothing.
    */
   persistTabAttribute: function TMP_ss_persistTabAttribute() {
      var aTab = gBrowser.mTabContainer.firstChild;
      if (aTab != gBrowser.mCurrentTab) {
         aTab.removeAttribute("selected");
         aTab.removeAttribute("flst_id");
      };

      try {
        /*
         * XUL Tab attributes to (re)store
         * Restored in nsSessionStore restoreHistory()
         */
         var _xulAttributes = gIsFirefox3 ? ["protected", "_locked", "fixed-label", "label-uri", "reload-data"] :
                                   ["image", "protected", "_locked", "fixed-label", "label-uri", "reload-data"];

         // make TreeStyleTab extension compatible with Tabmix Plus
         if ("TreeStyleTabBrowser" in window)
            _xulAttributes = _xulAttributes.concat(SessionData.tabTSTProperties);

         var ss = Cc["@mozilla.org/browser/sessionstore;1"].getService(Ci.nsISessionStore);
         _xulAttributes.forEach(function(aAttr) {
            ss.persistTabAttribute(aAttr);
         });
      } catch(ex) {
         tmLog("nsSessionStore could not add Attribute to persistTabAttribute: " + ex + "\n");
      }
   },

   /**
    * @brief         make sure that we don't enable both sessionStore and session manager
    *
    * @param msgNo   a Integer value - msg no. to show.
    *
    * @param start   a Boolean value - true if we call this function befor startup.
    *
    * @returns       Nothing.
    */
   setService: function TMP_ss_setSessionService(msgNo, start, win) {
      if(window.setSession || gPref.prefHasUserValue("extensions.tabmix.setDefault"))
         return;

     /*
      * From 2008-03-10 we don't set browser.sessionstore.enabled to false anymore
      * we use nsISessionStore service in TMP.
      * if we use TMP session manager we set all other sessionstore pref to false to disable SessionRestore
      *
      * Bug 449596  remove the browser.sessionstore.enabled pref
      * so here we don't set it to true, we just clear user pref to the default
      * if it the pref exist in firefox this set the pref to true
      * if the pref don't exist this will remove the pref
      */
      var TMP_sessionManager_enabled = gPref.getBoolPref("extensions.tabmix.sessions.manager") ||
                                     gPref.getBoolPref("extensions.tabmix.sessions.crashRecovery");
      if (!TMP_sessionManager_enabled) {
         if (gPref.prefHasUserValue("browser.sessionstore.enabled"))
            gPref.clearUserPref("browser.sessionstore.enabled");      
         return;
      }

      window.setSession = true;
      // if session manager extension is install disable TMP session manager
      if ("gSessionManager" in window) {
         // session manager extension take care about nsISessionStore service pref
         gPref.setBoolPref("extensions.tabmix.sessions.manager", false);
         gPref.setBoolPref("extensions.tabmix.sessions.crashRecovery", false);
      }
      else if (!this.getSsessionstoreEnabledPref()) {
         // TMP is the session manager disable all sessionstore pref
         // we only get here when user update from old tabmix version 
         // that set browser.sessionstore.enabled to false
         this.setSessionRestore(false);
      }
      else if (this.isSessionStoreEnabled()) {
         // ask the user to choose between TMP session manager and sessionstore
         var promptService = Cc["@mozilla.org/embedcomp/prompt-service;1"]
                                  .getService(Ci.nsIPromptService);
         var bundle_session = document.getElementById("bundle_session_manager");

         var msg = start ? bundle_session.getString("sm.disable.msg") + "\n\n" : "";
         msg += bundle_session.getString("sm.disable.msg" + msgNo);
         var buttonPressed = promptService.confirmEx(win || window,
                        "TabMix " + bundle_session.getString("sm.title"),
                        msg,
                        (promptService.BUTTON_TITLE_YES * promptService.BUTTON_POS_0)
                        + (promptService.BUTTON_TITLE_NO * promptService.BUTTON_POS_1),
                        null, null, null, null, {});

         if ((msgNo == 1 && buttonPressed == 1) || ((msgNo == 2 && buttonPressed == 0)))
            this.setSessionRestore(false);
         else {
            // we don't change any of sessionstore default setting
            // the user will be ask on exit what to do. (browser.warnOnRestart and browser.warnOnQuit are both true on default)
            gPref.setBoolPref("extensions.tabmix.sessions.manager", false);
            gPref.setBoolPref("extensions.tabmix.sessions.crashRecovery", false);
         }
      }
      // when user use firefox 2.0 profile browser.warnOnRestart and browser.warnOnQuit are both true on default
      else if (!gPref.prefHasUserValue("browser.warnOnRestart") ||
                !gPref.prefHasUserValue("browser.warnOnQuit ")) {
         gPref.setBoolPref("browser.warnOnRestart", false);
         gPref.setBoolPref("browser.warnOnQuit", false);
      }

      if (gPref.prefHasUserValue("browser.sessionstore.enabled"))
         gPref.clearUserPref("browser.sessionstore.enabled");

      delete window.setSession;
   },

   // Bug 449596  remove the browser.sessionstore.enabled pref
   getSsessionstoreEnabledPref: function () {
     if (gIsFirefox31)
       return true;

     return gPref.getBoolPref("browser.sessionstore.enabled");
   },

   isSessionStoreEnabled: function () {
     return this.getSsessionstoreEnabledPref() &&
        (gPref.getIntPref("browser.startup.page") == 3 || gPref.getBoolPref("browser.sessionstore.resume_from_crash"));
   },

   isAfterRestart: function () {
     if (this.getSsessionstoreEnabledPref()) {
       var ss = Cc["@mozilla.org/browser/sessionstartup;1"].
                      getService(Ci.nsISessionStartup);
       // when TMP session manager is enabled this is true only after restart
       return ss.doRestore();
     }
     return false;
   },

   setSessionRestore: function (aEnable) {
      gPref.setBoolPref("browser.warnOnRestart", aEnable);
      gPref.setBoolPref("browser.warnOnQuit", aEnable);
      gPref.setBoolPref("browser.sessionstore.resume_from_crash", aEnable);
      if (aEnable)
        gPref.setIntPref("browser.startup.page", 3);
      else if (gPref.getIntPref("browser.startup.page") == 3)
        gPref.setIntPref("browser.startup.page", 1);
   },

   /**
    * @brief           get attribute xultab data
    *
    * @param aTabData  an object value - tabData from nsSessionStore
    *
    * @param attrib    attribute name as string
    *
    * @returns         attribute value as string or empty string.
    */
   _getAttribute: function TMP_ss__getAttribute(aTabData, attrib) {
      // tabData.attributes is in use for Firefox 3.1+
//tmLog("", true);
      if (aTabData.attributes && attrib in aTabData.attributes)
         return aTabData.attributes[attrib];

      // restore attributes from the legacy Firefox 2.0/3.0 format
      if (aTabData.xultab) {
         var xultab = aTabData.xultab.split(" ");
         for ( var i= 0; i < xultab.length; i++ ){
            if (/^([^\s=]+)=(.*)/.test(xultab[i]) && RegExp.$1 == attrib)
               return decodeURI(RegExp.$2);
         }
      }
      return  "";
   }

}

var TMP_ClosedTabs = {
   _buttonBroadcaster: null,
   get buttonBroadcaster() {
      if (!this._buttonBroadcaster)
        this._buttonBroadcaster = document.getElementById("tmp_undocloseButton");
      return this._buttonBroadcaster;
   },

   /**
    * check to see if nsISessionStore is initialized
    */
   get ssIsON() {
      return "__SSi" in window;
   },

   // make btn_undoclose single-functionality or dual-functionality
   setButtonType: function(menuOnly) {
      var buttonType = menuOnly ? "menu" : "menu-button";
      if (this.buttonBroadcaster.getAttribute("type") != buttonType)
         this.buttonBroadcaster.setAttribute("type", buttonType);
   },

   setButtonDisableState: function ct_setButtonDisableState(aState) {
      if (typeof(aState) == "undefined")
         aState = this.count == 0;
      aState = aState ? "true" : "false";
      if (this.buttonBroadcaster.getAttribute("disabled") != aState)
         this.buttonBroadcaster.setAttribute("disabled", aState);
   },

   _ss: null,
   get ss() {
      if (!this._ss)
         this._ss = Cc["@mozilla.org/browser/sessionstore;1"].
                     getService(Ci.nsISessionStore);

      return this._ss;
   },

  /**
    * Get closed tabs count
    */
   get count() {
      if (this.ssIsON)
         return this.ss.getClosedTabCount(window);
      else
         return gBrowser.closedTabs.length;
   },

  /**
    * Get closed tabs data
    */
   get getClosedTabData() {
      if (this.ssIsON)
         return eval("(" + this.ss.getClosedTabData(window) + ")")
      else
         return gBrowser.closedTabs;
   },

   getUrl: function ct_getUrl(aTabData) {
      var history;
      if (this.ssIsON) {
         history = aTabData.state;
         var activeIndex = (history.index || history.entries.length) - 1;
         return history.entries[activeIndex].url;
      }
      history = aTabData.history;
      return history.getEntryAtIndex(history.index, false).URI.spec;
   },

  /**
   * @brief           update tab title from user name or bookmark.
   *
   * @param aTabData  an object value - tabData from nsSessionStore
   *
   * @param aUri      string value - url address
   *
   * @returns         tab title - string.
   */
   getTitle: function ct_getTitle(aTabData, aUri) {
      var data = this.ssIsON ? aTabData.state : {xultab: aTabData.properties};
      var fixedLabelUri = decodeURI(TMP_SessionStore._getAttribute(data, "label-uri"))
      if (fixedLabelUri == aUri || fixedLabelUri == "*")
         return decodeURI(TMP_SessionStore._getAttribute(data, "fixed-label"));

      return getTitleFromBookmark(aUri, this.ssIsON ? aTabData.title : aTabData.title);
   },

   /* .......... functions for closedtabs list menu and context menu .......... */

   populateUndoSubmenu: function ct_populateUndoSubmenu(aPopup) {
      if (isAfterCtrlClick(aPopup.parentNode))
         return false;

      beforeCommonList(aPopup);

      // populate menu
      var closedTabs = this.getClosedTabData;
      for (var i = 0; i < closedTabs.length; i++) {
         var m = document.createElement("menuitem");
         var tabData = closedTabs[i];
         // Grab the title and uri (make the uri friendly text)
         var url = this.getUrl(tabData);
         var title = this.getTitle(tabData, url);
         var _uri = makeURI(url);
         if ( _uri.scheme == "about" && title == "" )
            url = title = "about:blank";
         else try {
            url = _uri.scheme + ":\/\/" + _uri.hostPort + _uri.path;
         } catch (e) {
            url = title;
         }
         var label = title ? title : url;
         var count = (i + 1) + (i<10 ? " : " : ": ");
         m.setAttribute("label", count + label);
         m.setAttribute("tooltiptext", label);
         m.setAttribute("statustext", url);

         var image;
         if (this.ssIsON)
            image = gIsFirefox3 ? tabData.image : TMP_SessionStore._getAttribute(tabData.state, "image");
         else
            image = url.match(/^https/) ? null : tabData.image;
         if (image)
            m.setAttribute("image", image);
         m.setAttribute("class", "menuitem-iconic bookmark-item");
         m.setAttribute("value", i);
         if (i == 0)
            m.setAttribute("key", "key_undoCloseTab");
         aPopup.appendChild(m);
      }

      aPopup.appendChild(document.createElement("menuseparator"));
      // "Clear Closed Tabs List"
      var strings = document.getElementById("tmp-string-bundle");
      m = aPopup.appendChild(document.createElement("menuitem"));
      m.setAttribute("id", "clearClosedTabsList");
      m.setAttribute("label", strings.getString("undoclosetab.clear.label"));
      m.setAttribute("accesskey", strings.getString("undoclosetab.clear.accesskey"));
      m.setAttribute("value", -1);

      // "Open All in Tabs"
      m = aPopup.appendChild(document.createElement("menuitem"));
      m.setAttribute("id", "restoreAllClosedTabs");
      if (gIsFirefox3) {
         strings = gNavigatorBundle;
         m.setAttribute("label", strings.getString("menuOpenAllInTabs.label"));
         m.setAttribute("accesskey", strings.getString("menuOpenAllInTabs.accesskey"));
      }
      else {
        var bundleService = Cc["@mozilla.org/intl/stringbundle;1"].getService(Ci.nsIStringBundleService);
        var stringBundle = bundleService.createBundle("chrome://browser/locale/bookmarks/bookmarks.properties");
        m.setAttribute("label", stringBundle.GetStringFromName("cmd_bm_openfolder"));
        m.setAttribute("accesskey", stringBundle.GetStringFromName("cmd_bm_openfolder_accesskey"));
      }
      m.setAttribute("value", -2);

      return true;
   },

   checkForMiddleClick: function ct_checkForMiddleClick(aEvent) {
      if (aEvent.button != 1 || !tabxPrefs.getBoolPref("middleclickDelete"))
         return;

      var index = aEvent.originalTarget.value;
      if (index < 0)
         return;

      this.restoreTab('delete', index);
      var popup = aEvent.originalTarget.parentNode;
      if (this.count > 0)
         this.populateUndoSubmenu(popup);
      else {
         popup.hidePopup();
         if (popup.parentNode.id != "btn_undoclose")
            popup.parentNode.parentNode.hidePopup();
      }
   },

   addBookmarks: function ct_addBookmarks(index) {
      var tabData = this.getClosedTabData[index];
      var url = this.getUrl(tabData);
      var title = this.getTitle(tabData, url);
      if (gIsFirefox3)
         PlacesCommandHook.bookmarkLink(PlacesUtils.bookmarksMenuFolderId, url, title);
      else
         BookmarksUtils.addBookmark(url, title, undefined);
   },

   copyTabUrl: function ct_copyTabUrl(index) {
      var tab = this.getClosedTabData[index];
      var url = this.getUrl(tabData);
      var clipboard = Components.classes["@mozilla.org/widget/clipboardhelper;1"]
                      .getService(Components.interfaces.nsIClipboardHelper);

      clipboard.copyString(url);
   },

   restoreTab: function ct__restoreTab(aWhere, aIndex) {
      if (!this.ssIsON) {
         _restoreTab(aWhere, aIndex);
         return;
      }

      switch (aWhere) {
         case "window":
            this.SSS_restoreToNewWindow(aIndex);
            break;
         case "delete":
            this.getClosedTabAtIndex(aIndex);
            break;
/*
         case "bookmark":
            this.addBookmarks(aIndex);
            break;
         case "copyUrl":
            this.copyTabUrl(aIndex);
            break;
*/
         case "original":
            if (aIndex == -1) {
               this.getClosedTabAtIndex(aIndex);
               return;
            }
            else if (aIndex == -2) {
               this.SSS_restoerAllClosedTabs();
               return;
            }
            // else do the default
         default:
            this.SSS_undoCloseTab(aIndex, aWhere, true);
      }
   },

  /**
   * @brief           fetch the data of closed tab, while removing it from the array
   *
   * @param aIndex    a Integer value - 0 or grater index to remove
   *                  other value empty the list.
   *
   * @returns         close tab data at aIndex.
   */
   getClosedTabAtIndex: function ct_getClosedTabAtIndex(aIndex) {
      // update our session data
      // we don't use gBrowser.closedTabs when sessionStore is on
      var updateRDF = SessionManager.enableBackup && SessionPref.getBoolPref("save.closedtabs");
      if (updateRDF) {
        if (aIndex >= 0)
           SessionManager.deleteClosedtabAt(this.count - aIndex);
        else
           SessionManager.deleteWinClosedtabs(gThisWin);
      }

      var closedTabs = this.getClosedTabData;
      var closedTab;
      var state = { windows: [], _firstTabs: true };
      state.windows[0] = { _closedTabs: [] };
      // if aIndex is not > 0 we just past empy list to setWindowState
      // it's like remove all closed tabs from the list
      if (aIndex >= 0) {
         // purge closed tab at aIndex
         closedTab = closedTabs.splice(aIndex, 1).shift();
         state.windows[0]._closedTabs = closedTabs;
      }

      // replace existing _closedTabs
      this.ss.setWindowState(window, state.toSource(), false);

      this.setButtonDisableState();
      return closedTab;
   },

   SSS_restoreToNewWindow: function ct_restoreToNewWindow(aIndex) {
      var tabData = this.getClosedTabAtIndex(aIndex);
      //gBrowser.duplicateInWindow = function (aTab, aMoveTab, aTabData)
      // we past the current tab as reference to this window
      return gBrowser.duplicateInWindow(gBrowser.mCurrentTab, null, tabData);
   },

   SSS_restoerAllClosedTabs: function ct_SSS_restoerAllClosedTabs() {
      var closedTabCount = this.count;
      if (!_confirmOpenTabs(closedTabCount))
         return;

      this.setButtonDisableState(true);

      var aTab, blankTab;
      // catch blank tab
      var blankTabs = [];
      for (var i = 0; i < gBrowser.mTabs.length ; i++) {
         if (gBrowser.isBlankNotBusyTab(gBrowser.mTabs[i]))
            blankTabs.push(gBrowser.mTabs[i]);
      }

      for (i = 0; i < closedTabCount; i++) {
         blankTab = blankTabs.pop();
         this.SSS_undoCloseTab(0, "original", i==0, blankTab);
      }

      // remove unused blank tabs
      while(blankTabs.length > 0){
         blankTab = blankTabs.pop();
         gBrowser.removeTab(blankTab);
      }

      gBrowser.mTabContainer.nextTab = 1;
   },

   SSS_undoCloseTab: function ct_SSS_undoCloseTab(aIndex, aWhere, aSelectRestoredTab, aTabToRemove) {
      if (!tabxPrefs.getBoolPref("undoClose") || this.count == 0)
         return null;

      // get tab data
      var tabData = this.getClosedTabAtIndex(aIndex);

      var cTab = gBrowser.mCurrentTab;
      if (aWhere == "current") {
         cTab.collapsed = true;
         aTabToRemove = cTab;
         tabData.pos = cTab._tPos;
      }
      else if (typeof(aTabToRemove) == "undefined" && gBrowser.isBlankNotBusyTab(cTab))
         aTabToRemove = cTab;

      // add restored tab to current window
      var newTab;
      if (gIsFirefox3) {
         newTab = gBrowser.addTab();
         this.ss.setTabState(newTab, tabData.state.toSource());
      }
      else { // there is no setTabState/getTabState in firefox 2.0
        var state = { windows: [ { tabs: [tabData.state] } ] };
        this.ss.setWindowState(window, state.toSource(), false);
        newTab = gBrowser.getTabForLastPanel();
      }

      // after we open new tab we only need to fix position if this is true
      var restorePosition = tabxPrefs.getBoolPref("undoClosePosition");
      if ( aWhere == "current" || (aWhere == "original" && restorePosition) ) {
         gBrowser.TMmoveTabTo(newTab, Math.min(gBrowser.mTabs.length - 1, tabData.pos), 1);
      }
      else if (aWhere != "end")
         // we don't call TMP_openTabNext from add tab if it called from sss_undoCloseTab
         gBrowser.TMP_openTabNext(newTab);

      if (aSelectRestoredTab) {
         content.focus();
         gBrowser.TMP_selectNewForegroundTab(newTab, false, null, false);
      }

      if (aTabToRemove)
         gBrowser.removeTab(aTabToRemove);

      return newTab;
   },

   TMP_undoCloseTab: function ct_TMP_undoCloseTab() {
      if( !tabxPrefs.getBoolPref("undoClose" ) || !gBrowser.closedTabs.length )
         return null;

      var tabData = getClosedTab("original", 0);
      return gBrowser.restoreTab(tabData.pos, tabData.history, tabData.properties, tabData.scroll);
   },

   undoCloseTab: function ct_undoCloseTab(aIndex, aWhere) {
      var newTab;
      if (this.ssIsON)
         newTab = this.SSS_undoCloseTab(aIndex || 0, aWhere || "original", true);
      else
         newTab = this.TMP_undoCloseTab();

      return newTab;
   }

}

var convertSession = {
   get _prompt() {
      return Cc["@mozilla.org/embedcomp/prompt-service;1"]
                            .getService(Ci.nsIPromptService);
   },

   get getTitle() {
      return document.getElementById("tmp-string-bundle").getString("incompatible.title") + " - " +
                document.getElementById("bundle_session_manager").getString("sm.title");
   },

   getString: function cs_getString(aEntity) {
      return document.getElementById("bundle_session_manager").
         getString("sm.extension.convert." + aEntity);
   },

   startup: function cs_startup() {
      var isFirstWindow = "gSessionManager" in window && numberOfWindows() == 1 && !window.duplicat;
      if (!isFirstWindow)
         return;

      var sessions = SessionManager.getSessionList();
      if (!sessions)
         return;

      if(SessionManager.nodeHasArc("rdf:gSessionManager", "status"))
         return;

      SessionManager.setLiteral("rdf:gSessionManager", "status", "converted");
      SessionManager.saveStateDelayed();

      var rv = this.confirm(this.getString("msg1") + "\n\n" + this.getString("msg2"));
      if (rv == 0)
         this.doConvert(sessions);
   },

   selectFile: function cs_selectFile(aWindow) {
      const nsIFilePicker = Ci.nsIFilePicker;
      var fp = Cc["@mozilla.org/filepicker;1"].createInstance(nsIFilePicker);

      fp.init(aWindow, this.getString("selectfile"), nsIFilePicker.modeOpen);
      fp.defaultString="session.rdf";
      fp.appendFilter(this.getString("rdffiles"), "*.rdf");
      fp.appendFilter(this.getString("sessionfiles"), "*session*.*");
      fp.appendFilters(nsIFilePicker.filterText | nsIFilePicker.filterAll);

      if (fp.show() != nsIFilePicker.returnOK)
         return;

      this.convertFile(fp.fileURL.spec);
   },

   convertFile: function cs_convertFile(aFileUri) {
      var sessions;
      if (aFileUri) {
         try {
            var tmpDATASource = SessionManager.DATASource;
            SessionManager.DATASource = SessionManager.RDFService.GetDataSourceBlocking(aFileUri);
            sessions = SessionManager.getSessionList();
         } catch (e) { // corrupted session.rdf
            SessionManager.DATASource = tmpDATASource;
            tmLog(e);
         }
      }
      else
         sessions = SessionManager.getSessionList();

      var msg;
      if (!sessions) {
         this._prompt.alert(null, this.getTitle, this.getString("nosessions"));
         return;
      }
      var rv = 0;
      if(SessionManager.nodeHasArc("rdf:gSessionManager", "status")) {
         rv = this.confirm(this.getString("alreadyconverted") + "\n\n" + this.getString("doitagain"));
      }
      else {
         SessionManager.setLiteral("rdf:gSessionManager", "status", "converted");
         SessionManager.saveStateDelayed();
      }
      if (rv == 0)
         this.doConvert(sessions);

      if (tmpDATASource)
         SessionManager.DATASource = tmpDATASource;
   },

   confirm: function cs_confirm(aMsg) {
      var promptService = this._prompt;
      return promptService.confirmEx(null,
                           this.getTitle,
                           aMsg,
                           (promptService.BUTTON_TITLE_YES * promptService.BUTTON_POS_0)
                           + (promptService.BUTTON_TITLE_NO * promptService.BUTTON_POS_1),
                           null, null, null, null, {});
   },

   doConvert: function cs_doConvert(sessions) {
      var sessionsPath = sessions.path.push(gSessionPath[3]);
      var sessionsName = sessions.list.push("Crashed Session");
      var _count = 0;
      this.sessionList = [];
      for (var i = 0; i < sessions.path.length; i++ ) {
         var sessionState = this.getSessionState(sessions.path[i]);

         // get timestamp from nameExt property
         var dateString = "", fileDate;
         var nameExt = SessionManager.getLiteralValue(sessions.path[i], "nameExt");
         if (nameExt) {
            var date = nameExt.substr(nameExt.length - 20, 10);
            var time = nameExt.substr(nameExt.length - 9, 8);
            fileDate = " (" + date.split("/").join("-") + ")";
            dateString = " (" + date.split("/").join("-") + " " + time.substr(0, time.length - 3) + ")";
            var _time = time.split(":");
            var timestamp = new Date(date).valueOf() + 3600*_time[0] + 60*_time[1] + 1*_time[2];
         }
         var sessionName = unescape(sessions.list[i]);
         var name = "[ Tabmix ] " + sessionName + dateString;
         var fileName = gSessionManager.makeFileName("Tabmix - " + sessionName + fileDate);

          if (this.save(sessionState, timestamp, name, fileName)) {
            _count++;
            var item = { name: name, fileName: fileName };
            this.sessionList.push(item);
          }
      }

      var msg;
      if (_count == 0) {
         this._prompt.alert(null, this.getTitle, this.getString("unable"));
         return;
      }
      else if (_count > 1)
         msg = _count + " " + this.getString("many") + "\n" + this.getString("rename");
      else
         msg = this.getString("one") + "\n" + this.getString("rename");

      var sessionToDelete = gSessionManager.selectSession(msg, gSessionManager._string("remove_session_ok"),
                                                   { multiSelect: true, remove: true },
                                                   function() { return convertSession.sessionList;});
      delete this.sessionList;
      if (sessionToDelete)
         gSessionManager.remove(sessionToDelete);
   },

   save: function cs_save(aSession, aTimestamp, aName, aFileName) {
      if (aSession.windows.length == 0)
         return false;

      if (!aSession.session)
         aSession.session = { state:"stop" };
      var oState = "[SessionManager]\nname=" + aName + "\ntimestamp=" + aTimestamp + "\n" + aSession.toSource();
      var file = gSessionManager.getSessionDir(gSessionManager.makeFileName(aName));
      try {
         file = gSessionManager.getSessionDir(aFileName, true);
         gSessionManager.writeFile(file, oState);
      }
      catch (ex) {
         tmLog(ex);
         return false;
      }
      return true;
   },

   getSessionState: function cs_getSessionState(aPath) {
      var _windows = [];
      var sessionEnum = SessionManager.initContainer(aPath).GetElements();
      while (sessionEnum.hasMoreElements()) {
         var rdfNodeWindow = sessionEnum.getNext();
         if (rdfNodeWindow instanceof tmRDFResource) {
            var windowPath = rdfNodeWindow.QueryInterface(tmRDFResource).Value;
            if (SessionManager.nodeHasArc(windowPath, "dontLoad"))
               continue;
            var aWindowState = this.getWindowState(rdfNodeWindow);
            if (aWindowState) // don't save empty windows
               _windows.push(aWindowState);
         }
      }
      return { windows: _windows };
   },

   getWindowState: function cs_getWindowState(rdfNodeWindow) {
      var state = { tabs: [], selected: 0, _closedTabs: [] };

      var rdfNodeTabs = SessionManager.getResource(rdfNodeWindow, "tabs");
      if (!(rdfNodeTabs instanceof tmRDFResource) || SessionManager.containerEmpty(rdfNodeTabs)) {
         return null;
      }
      state.tabs = this.getTabsState(rdfNodeTabs);
      state._closedTabs = this.getClosedTabsState(SessionManager.getResource(rdfNodeWindow, "closedtabs"));
      state.selected = SessionManager.getIntValue(rdfNodeWindow, "selectedIndex");
      return state;
   },

   getTabsState: function cs_getTabsState(rdfNodeTabs) {
      var _tabs = [];
      var tabsEnum = SessionManager.initContainer(rdfNodeTabs).GetElements();
      while (tabsEnum.hasMoreElements()) {
         var rdfNodeTab = tabsEnum.getNext();
         if (rdfNodeTab instanceof tmRDFResource)
            _tabs.push(this.getTabState(rdfNodeTab));
      }
      return _tabs;
   },

   getClosedTabsState: function cs_getClosedTabsState(rdfNodeTabs) {
      var _tabs = [];
      var tabsEnum = SessionManager.initContainer(rdfNodeTabs).GetElements();
      while (tabsEnum.hasMoreElements()) {
         var rdfNodeTab = tabsEnum.getNext();
         if (rdfNodeTab instanceof tmRDFResource) {
            var closedTab = {};
            closedTab.state = this.getTabState(rdfNodeTab);
            closedTab.title = closedTab.state.entries[closedTab.state.index - 1].title;
            if (gIsFirefox3)
               closedTab.image = SessionManager.getLiteralValue(rdfNodeTab, "image");
            closedTab.pos = SessionManager.getIntValue(rdfNodeTab, "tabPos");
            // we use in the RDF revers order
            _tabs.unshift(closedTab);
         }
      }
      return _tabs;
   },

   getTabState: function cs_getTabState(rdfNodeTab) {
      var tabData = {entries:[], index: 0, zoom: 1, disallow:"", extData: null, text:""};
      tabData.entries = this.getHistoryState(rdfNodeTab);
      tabData.index = SessionManager.getIntValue(rdfNodeTab, "index") + 1;
      tabData.zoom = SessionManager.getLiteralValue(rdfNodeTab, "scroll").split(",")[2];
      var properties = SessionManager.getLiteralValue(rdfNodeTab, "properties");
      var tabAttribute = ["Images","Subframes","MetaRedirects","Plugins","Javascript"];

      var booleanAttrLength = SessionData.tabAttribute.length + SessionData.docShellItems.length;
      var tabProperties = properties.substr(0, booleanAttrLength);
      var disallow = [];
      for (var j = 0; j < tabAttribute.length; j++ ) {
         if (tabProperties.charAt(j+2) != "1")
            disallow.push(tabAttribute[j]);
      }
      tabData.disallow = disallow.join(",");
      var xultab = [];
      if (!gIsFirefox3) {
         var image = SessionManager.getLiteralValue(rdfNodeTab, "image");
         if (image)
            xultab.push("image=" + image);
      }
      // xultab replace in fireofx 3.1+ with tabData.attributes
      // but nsSessionStore can still read xultab
      if (gIsFirefox31) {
         tabData.attributes = {};
         if (tabProperties.charAt(0) == "1" && properties.indexOf("protected=") == -1)
            tabData.attributes["protected"] = "true";
         if (properties.indexOf("_locked=") == -1)
            tabData.attributes["_locked"] = (tabProperties.charAt(1) == "1");

         if (properties.length > booleanAttrLength) {
            properties = properties.substr(booleanAttrLength + 1).split(" ");
            properties.forEach(function(aAttr) {
              if (/^([^\s=]+)=(.*)/.test(aAttr)) {
                tabData.attributes[RegExp.$1] = RegExp.$2;
              }
            });
         }
      }
      else {
         if (tabProperties.charAt(0) == "1" && properties.indexOf("protected=") == -1)
            xultab.push("protected=true");
         if (properties.indexOf("_locked=") == -1)
            xultab.push("_locked=" + (tabProperties.charAt(1) == "1"));
         tabData.xultab = xultab.join(" ");
         if (properties.length > booleanAttrLength)
           tabData.xultab += properties.substr(booleanAttrLength + 1);
      }
      return tabData;
   },

   getHistoryState: function cs_getHistoryState(rdfNodeTab) {
      var tmpData = SessionManager.getLiteralValue(rdfNodeTab, "history").split("|-|");
      var sep = tmpData.shift(); // remove seperator from data
      var historyData = tmpData.join("|-|").split(sep);
      var historyCount = historyData.length/3;
      var entries = [];
      for ( var i = 0; i < historyCount; i++ ){
         var entry = { url:"", children:[], ID: 0};
         var index = i * 3;
         entry.url = historyData[index + 1];
         entry.title = unescape(historyData[index]);
         entry.scroll = historyData[index + 2];
         entries.push(entry);
      }
      return entries;
   }
}
