/*
    The IP location database is in the format:
        [4 byte start IP 1][4 byte start IP 2]... [2 byte country code 1][2 byte country code 2]...
        (all IPs from 0x000000 to 0xFFFFFF must exist, with gaps listed as code "??")
*/

var IPDB_stream = null;
var IPDB_binary = null;
var IPDB_length = 0;

function Flagfox_loadIPDB()
{
    const id = "{1018e4d6-728f-4b20-ad56-37578a4de76b}";
    var file = Components.classes["@mozilla.org/extensions/manager;1"]
                         .getService(Components.interfaces.nsIExtensionManager)
                         .getInstallLocation(id)
                         .getItemLocation(id);
    file.append("db.dat");
    if (!file.exists())
        throw "IPDB file not found (" + file.path + ")";

    // Open an input stream for file
    // (large JavaScript arrays are slow bloated memory wasters; 30MB+ is ridiculous)
    IPDB_stream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                            .createInstance(Components.interfaces.nsIFileInputStream)
                            .QueryInterface(Components.interfaces.nsISeekableStream);
    IPDB_stream.init(file, 0x01, 0444, 0);  // read-only, read by owner/group/others, normal behavior
    if (!IPDB_stream.available())
        throw "IPDB file failed to load (" + file.path + ")";

    // Reading binary needs a helper interface
    IPDB_binary = Components.classes["@mozilla.org/binaryinputstream;1"]
                            .createInstance(Components.interfaces.nsIBinaryInputStream);
    IPDB_binary.setInputStream(IPDB_stream);

    IPDB_length = IPDB_stream.available() / 6;   // 4 bytes for IP; 2 bytes for country code
    if (IPDB_length < 80000 || Math.floor(IPDB_length) != IPDB_length)
        throw "IPDB file is corrupt (loaded bytes: " + IPDB_stream.available() + ")";
}

function Flagfox_closeIPDB()
{
    if (IPDB_stream != null)
    {
        IPDB_stream.close();
        IPDB_stream = null;
        IPDB_binary = null;
        IPDB_length = 0;
    }
}

function Flagfox_lookupIP(ipstring)
{
    function getEntry(index)  // Reading past the end will throw an exception
    {
        IPDB_stream.seek(0,index*4);
        return new Array( IPDB_binary.read32(), IPDB_binary.read32() );  // [start IP, end IP]
    }

    function readCountry()  // Finds country code based on current position via last IPDB_getEntry() call
    {
        const IPblock_length = IPDB_length*4;
        IPDB_stream.seek(0,IPblock_length + (IPDB_stream.tell()-8)/2);
        return String.fromCharCode( IPDB_binary.read8(), IPDB_binary.read8() );  // Country code (2 char string)
    }

    function dottedquad2long(dottedquad)
    {
        const bytes = dottedquad.split("\.");
        return (16777216 * parseInt(bytes[0])) + (65536 * parseInt(bytes[1])) + (256 * parseInt(bytes[2])) + parseInt(bytes[3]);
    }

    function binarySearch(x, low, high)
    {
        if (low > high)
            throw null;

        var middle = Math.floor((low + high) / 2);
        var entry = getEntry(middle);

        if (entry[0] <= x && x < entry[1])  // entry[1] is start of next range in list; not part of this range
            return entry;
        else if (x < entry[0])
            return binarySearch(x, low, middle-1);
        else  // x >= entry[1]
            return binarySearch(x, middle+1, high);
    }

    if (ipstring == "")
        return null;
    try
    {
        binarySearch(dottedquad2long(ipstring), 0, IPDB_length);
        var countrycode = readCountry();
        if (countrycode != "??")  // countrycode "??" indicates a gap in the IPDB
            return new Array( countrycode, countrynames.getString(countrycode) );  // [country code, country name]
    } catch (e) {}
    return null;
}
