
// $Id: converterOverlay.js,v 1.44 2008/08/03 00:19:48 bogdan Exp $

/* ***** BEGIN LICENSE BLOCK ***** 
 * ------------------------ Contributor(s) ------------------------------------
 *      Copyright 2004-2007 Bogdan Stancescu <converter@moongate.ro>
 * ----------------------------------------------------------------------------
 *
 * Parts of the extension's basic groundwork (install.rdf, etc) copied from
 *   Jaap Haitsma's Dictionarysearch extension
 *   (http://dictionarysearch.mozdev.org/)
 *
 * ----------------------------------------------------------------------------
 *
 * This work is licensed under the Creative Commons
 * Attribution-No Derivative Works 3.0 Unported license
 *
 *
 * For details on the Creative Commons license, please see
 * http://creativecommons.org/licenses/by-nd/3.0/
 *
 * ----------------------------------------------------------------------------
 *
 * The Creative Commons license in short (the text below is for informational
 * purposes only -- please see the URL above for the legally binding license):
 *
 * You are free to Share - to copy, distribute and transmit the work
 * under the following conditions:
 *
 * +  Attribution. You must attribute the work in the manner specified by
 * the author or licensor (but not in any way that suggests that they
 * endorse you or your use of the work).
 *
 * +  No Derivative Works. You may not alter, transform, or build upon
 * this work.
 *
 * For any reuse or distribution, you must make clear to others the license
 * terms of this work. The best way to do this is with a link to this web page.
 * Any of the above conditions can be waived if you get permission from the
 * copyright holder.
 *
 * Nothing in this license impairs or restricts the author's moral rights.
 *
 * Representations, Warranties and Disclaimer
 * UNLESS OTHERWISE MUTUALLY AGREED TO BY THE PARTIES IN WRITING,
 * LICENSOR OFFERS THE WORK AS-IS AND MAKES NO REPRESENTATIONS OR
 * WARRANTIES OF ANY KIND CONCERNING THE WORK, EXPRESS, IMPLIED,
 * STATUTORY OR OTHERWISE, INCLUDING, WITHOUT LIMITATION,
 * WARRANTIES OF TITLE, MERCHANTIBILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * NONINFRINGEMENT, OR THE ABSENCE OF LATENT OR OTHER DEFECTS,
 * ACCURACY, OR THE PRESENCE OF ABSENCE OF ERRORS, WHETHER OR NOT DISCOVERABLE.
 * SOME JURISDICTIONS DO NOT ALLOW THE EXCLUSION OF IMPLIED WARRANTIES,
 * SO SUCH EXCLUSION MAY NOT APPLY TO YOU.
 *
 * Limitation on Liability
 * EXCEPT TO THE EXTENT REQUIRED BY APPLICABLE LAW, IN NO EVENT WILL
 * LICENSOR BE LIABLE TO YOU ON ANY LEGAL THEORY FOR ANY SPECIAL, INCIDENTAL,
 * CONSEQUENTIAL, PUNITIVE OR EXEMPLARY DAMAGES ARISING OUT OF THIS LICENSE
 * OR THE USE OF THE WORK, EVEN IF LICENSOR HAS BEEN ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * ***** END LICENSE BLOCK ***** */

/*
VERSION: 0.7.1

TO DO:
 - metric weights => imperial weights
 - surfaces
 - volumes
 - update example page with speeds plus everything above
 - example page: adaptive results -- in2m, m2in, kmph2miph, miph2kmph, lb2kg
*/

/*
  This string defines which unit is parsed by what converter.
  
  Notes:
  1. Since we separate units when parsing the selection by spaces,
     it is safe to also separate them by spaces here. Therefore the format
     of this string is ((unit)(<space>)*(converter)(<space>)*)*
  2. The converters are functions, defined below. In order to avoid name clashes
     between the converter functions and other functions in Mozilla, we
     programatically prepend "converter_" to the name of the converter defined
     here when we evaluate.
  3. Make sure there is a backslash at the end of every line in this string.
     A backslash tells the JavaScript interpreter the line is *continued*
     on the next line (i.e. the lines are simply concatenated in the final
     string, no line breaks result in the end). Please note this behaviour is
     variable in JavaScript depending on the browser; luckily we don't have to
     worry about that in this environment. :)
  4. The selection parser is case-insensitive, so all units in here *MUST* be
     in small letters. For instance "C" as a unit in the string below
     would NOT match "15 C" in the selection, but "c" does.
*/

var conv_table=new String("\
  \"               in2m \
  in               in2m \
  inch             in2m \
  inches           in2m \
  -inch            in2m \
  -inches          in2m \
  \
  mil              mil2m \
  \
  mi               mi2m \
  mile             mi2m \
  miles            mi2m \
  -mile            mi2m \
  -miles           mi2m \
  \
  ft               ft2m \
  foot             ft2m \
  feet             ft2m \
  -foot            ft2m \
  -feet            ft2m \
  \'               ft2m \
  \
  yard             yd2m \
  yards            yd2m \
  -yard            yd2m \
  -yards           yd2m \
  yd               yd2m \
  \
  ounce            oz2kg \
  oz               oz2kg \
  ounces           oz2kg \
  \
  pound            lb2kg \
  pounds           lb2kg \
  -pound           lb2kg \
  -pounds          lb2kg \
  lb               lb2kg \
  lbs              lb2kg \
  \
  stone            stone2kg \
  stones           stone2kg \
  \
  kg               kg2lb \
  kgs              kg2lb \
  kilogram         kg2lb \
  kilograms        kg2lb \
  \
  g                g2oz \
  gram             g2oz \
  grams            g2oz \
  \
  m                m2in \
  meter            m2in \
  meters           m2in \
  metre            m2in \
  metres           m2in \
  \
  mm               mm2in \
  millimetre       mm2in \
  millimetres      mm2in \
  millimeter       mm2in \
  millimeters      mm2in \
  \
  km               km2in \
  kilometre        km2in \
  kilometres       km2in \
  kilometer        km2in \
  kilometers       km2in \
  \
  "+String.fromCharCode(181)+"m               microm2in \
  micrometre       microm2in \
  micrometres      microm2in \
  micrometer       microm2in \
  micrometers      microm2in \
  micron           microm2in \
  microns          microm2in \
  \
  cm               cm2in \
  centimetre       cm2in \
  centimetres      cm2in \
  centimeter       cm2in \
  centimeters      cm2in \
  \
  "+String.fromCharCode(176)+"celsius         c2f \
  "+String.fromCharCode(176)+"celcius         c2f \
  "+String.fromCharCode(176)+"c               c2f \
  "+String.fromCharCode(176)+"_celsius        c2f \
  "+String.fromCharCode(176)+"_celcius        c2f \
  "+String.fromCharCode(176)+"_c              c2f \
  \u00BAcelsius         c2f \
  \u00BAcelcius         c2f \
  \u00BAc               c2f \
  \u00BA_celsius        c2f \
  \u00BA_celcius        c2f \
  \u00BA_c              c2f \
  "+ // DO NOT change the order here! Moving 'degree_c' up will kill the other matches!
  "\
  c                c2f \
  celsius          c2f \
  celcius          c2f \
  centigrade       c2f \
  centigrades      c2f \
  degree_celsius   c2f \
  degrees_celsius  c2f \
  degree_centigrade  c2f \
  degrees_centigrade c2f \
  degree_c         c2f \
  degrees_c        c2f \
  \
  "+String.fromCharCode(176)+"f               f2c \
  "+String.fromCharCode(176)+"fahrenheit      f2c \
  "+String.fromCharCode(176)+"_fahrenheit     f2c \
  "+String.fromCharCode(176)+"_f              f2c \
  \u00BAf               f2c \
  \u00BAfahrenheit      f2c \
  \u00BA_fahrenheit     f2c \
  \u00BA_f              f2c \
  f                  f2c \
  fahrenheit         f2c \
  degree_fahrenheit  f2c \
  degrees_fahrenheit f2c \
  degree_f           f2c \
  degrees_f          f2c \
  \
  mph              miph2kmph \
  mi/h             miph2kmph \
  mile/h           miph2kmph \
  miles/h          miph2kmph \
  mile/hour        miph2kmph \
  miles/hour       miph2kmph \
  miles_per_hour   miph2kmph \
  mile_per_hour    miph2kmph \
  mile_an_hour     miph2kmph \
  miles_an_hour    miph2kmph \
  \
  km/h             kmph2miph \
  kmph             kmph2miph \
  kilometers/hour  kmph2miph \
  kilometres/hour  kmph2miph \
  kilometers_per_hour kmph2miph \
  kilometres_per_hour kmph2miph \
  kilometer_per_hour kmph2miph \
  kilometre_per_hour kmph2miph \
  \
  degree           deg2all \
  degrees          deg2all \
  deg              deg2all \
  "+String.fromCharCode(176)+"            deg2all \
  \
  m/s              mps2all \
  mps              mps2all \
  meters_per_second mps2all \
  metres_per_second mps2all \
  meter_per_second mps2all \
  metre_per_second mps2all \
  \
  l                litre2gallon \
  liter            litre2gallon \
  liters           litre2gallon \
  litre            litre2gallon \
  litres           litre2gallon \
  \
  gallon           gallon2litre \
  gallons          gallon2litre \
  gal              gallon2litre \
  \
  fluid_ounce      floz2ml \
  fluid_ounces     floz2ml \
  fl._oz.          floz2ml \
  fl._oz           floz2ml \
  fl_oz.           floz2ml \
  fl.oz.           floz2ml \
  fl_oz            floz2ml \
  floz             floz2ml \
  fluidounce       floz2ml \
  fluidounces      floz2ml \
  \
  ml               ml2floz \
  milliliter       ml2floz \
  milliliters      ml2floz \
  mililiter        ml2floz \
  mililiters       ml2floz \
  \
  pa               pa2psi \
  pascal           pa2psi \
  pascals          pa2psi \
  kpa              kpa2psi \
  psi              psi2kpa \
  \
  internal_time    internal_time \
  \
  square_foot      sqft2sqm \
  square_feet      sqft2sqm \
  square_ft        sqft2sqm \
  sq_ft            sqft2sqm \
  sq._ft.          sqft2sqm \
  sq_ft.           sqft2sqm \
  sq._ft           sqft2sqm \
  sq_feet          sqft2sqm \
  sq_foot          sqft2sqm \
  sf               sqft2sqm \
  foot"+String.fromCharCode(178)+" sqft2sqm \
  feet"+String.fromCharCode(178)+" sqft2sqm \
  ft"+String.fromCharCode(178)+" sqft2sqm \
  ft^2             sqft2sqm \
  foot^2           sqft2sqm \
  feet^2           sqft2sqm \
  \
  square_meters    sqm2sqft \
  square_metres    sqm2sqft \
  square_meter     sqm2sqft \
  square_metre     sqm2sqft \
  m"+String.fromCharCode(178)+" sqm2sqft \
  meter"+String.fromCharCode(178)+" sqm2sqft \
  metre"+String.fromCharCode(178)+" sqm2sqft \
  meters"+String.fromCharCode(178)+" sqm2sqft \
  metres"+String.fromCharCode(178)+" sqm2sqft \
  m^2              sqm2sqft \
  meter^2          sqm2sqft \
  metre^2          sqm2sqft \
  meters^2         sqm2sqft \
  metres^2         sqm2sqft \
  \
  cc               cc2cuin \
  cm^3             cc2cuin \
  cm"+String.fromCharCode(179)+" cc2cuin \
  cu_in            cuin2cc \
  in^3             cuin2cc \
  in"+String.fromCharCode(179)+" cuin2cc \
  \
  mpg              mpg2lptm \
  miles_per_gallon mpg2lptm \
  \
  nm               nm2ftlb \
  newton_metre     nm2ftlb \
  newton_metres    nm2ftlb \
  newton_meter     nm2ftlb \
  newton_meters    nm2ftlb \
");

/*

  ====================================================================
                          THE CONVERTERS
  ====================================================================
  
  Ok, now that we know which unit is parsed with what converter, let's
  define the converters themselves. If you add extra converters, please
  make sure the order of the functions below is the same as in the string
  above.
  
*/

function converter_in2m(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.0254;
  result.inUnit="in";
  result.outUnit="m";
  return(result);
}

function converter_mil2m(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.0000254;
  result.inUnit="mil";
  result.outUnit="m";
  return(result);
}

function converter_mi2m(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*1609.344;
  result.inUnit="miles";
  result.outUnit="m";
  return(result);
}

function converter_ft2m(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.3048;
  result.inUnit="ft";
  result.outUnit="m";
  return(result);
}

function converter_yd2m(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.9144;
  result.inUnit="yd";
  result.outUnit="m";
  return(result);
}

function converter_oz2kg(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.0283;
  result.inUnit="oz";
  result.outUnit="kg";
  return(result);
}

function converter_lb2kg(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.45359237;
  result.inUnit="lbs";
  result.outUnit="kg";
  return(result);
}

function converter_stone2kg(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*6.35029318;
  result.inUnit="stone";
  result.outUnit="kg";
  return(result);
}

function converter_kg2lb(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/0.45359237;
  result.inUnit="kg";
  result.outUnit="lbs";
  return(result);
}

function converter_g2oz(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/28.3;
  result.inUnit="g";
  result.outUnit="oz";
  return(result);
}

function converter_m2in(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*1000/25.4;
  result.inUnit="m";
  result.outUnit="in";
  return(result);
}

function converter_mm2in(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/25.4;
  result.inUnit="mm";
  result.outUnit="in";
  return(result);
}

function converter_km2in(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=(v/0.0254)*1000;
  result.inUnit="km";
  result.outUnit="in";
  return(result);
}

function converter_microm2in(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/25400;
  result.inUnit=String.fromCharCode(181)+"m";
  result.outUnit="in";
  return(result);
}

function converter_cm2in(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/2.54;
  result.inUnit="cm";
  result.outUnit="in";
  return(result);
}

function converter_c2f(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=Math.round((v*1.8+32)*10)/10;
  result.inUnit=String.fromCharCode(176)+"C";
  result.outUnit=String.fromCharCode(176)+"F";
  return(result);
}

function converter_f2c(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=Math.round(10*(v-32)/1.8)/10;
  result.inUnit=String.fromCharCode(176)+"F";
  result.outUnit=String.fromCharCode(176)+"C";
  return(result);
}

function converter_miph2kmph(v)
{
  var result=new Object();
  result.inValue=v;
  if (v<2) {
    result.outValue=Math.round(v*16093.44)/10000;
  } else if (v < 10) {
    result.outValue=Math.round(v*160.9344)/100;
  } else {
    result.outValue=Math.round(v*1.609344);
  }
  result.inUnit="mph";
  result.outUnit="km/h";
  return(result);
}

function converter_kmph2miph(v)
{
  var result=new Object();
  result.inValue=v;
  if (v<3) {
    result.outValue=Math.round(v/0.0001609344)/10000;
  } else if (v<15) {
    result.outValue=Math.round(v/0.01609344)/100;
  } else {
    result.outValue=Math.round(v/1.609344);
  }
  result.inUnit="km/h";
  result.outUnit="mph";
  return(result);
}

function converter_deg2all(v)
{
  var result=new Object();
  result.inValue=v;
  if (gConverterPrefs.getPref('pref_metric')) {
    result.inUnit=String.fromCharCode(176)+"F";
    result.outUnit=String.fromCharCode(176)+"C";
    result.outValue=(v-32)/1.8;
  } else {
    result.inUnit=String.fromCharCode(176)+"C";
    result.outUnit=String.fromCharCode(176)+"F";
    result.outValue=v*1.8+32;
  }
  return(result);
}

function converter_mps2all(v)
{
  var result=new Object();
  result.inValue=v;
  result.inUnit="m/s";
  if (gConverterPrefs.getPref('pref_metric')) {
    result.outUnit="km/h";
    result.outValue=v * 18 / 5;
  } else {
    result.outUnit="mph";
    result.outValue=v * 3600 / 1610.3;
  }
  return(result);
}

function converter_litre2gallon(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/3.785411784;
  result.inUnit="L";
  result.outUnit="gal";
  return(result);
}

function converter_gallon2litre(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*3.785411784;
  result.inUnit="gal";
  result.outUnit="L";
  return(result);
}

function converter_ml2floz(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/29.5735295625;
  result.inUnit="ml";
  result.outUnit="fl oz";
  return(result);
}

function converter_floz2ml(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*29.5735295625;
  result.inUnit="fl oz";
  result.outUnit="ml";
  return(result);
}

function converter_pa2psi(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/6894.75729;
  result.inUnit="Pa";
  result.outUnit="psi";
  return(result);
}

function converter_kpa2psi(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/6.89475729;
  result.inUnit="KPa";
  result.outUnit="psi";
  return(result);
}

function converter_psi2kpa(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*6.89475729;
  result.inUnit="psi";
  result.outUnit="KPa";
  return(result);
}

function converter_cc2cuin(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/16.387064;
  result.inUnit="cm"+String.fromCharCode(179);
  result.outUnit="in"+String.fromCharCode(179);
  return(result);
}

function converter_cuin2cc(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*16.387064;
  result.inUnit="in"+String.fromCharCode(179);
  result.outUnit="cm"+String.fromCharCode(179);
  return(result);
}

function converter_mpg2lptm(v)
{
  var result=new Object();
  result.only_show_converted=true;
  result.not_numerical=true;
  var oU=new Number(235.2/v);
  var oI=new Number(282.5/v);
  result.inValue=v;
  result.inUnit="mpg";
  // var out=out.toFixed(-val_e+2);
  result.outCustom=v+"mp(US)g = "+oU.toFixed(2)+" L/100 km; "+v+" mp(Imp)g = "+oI.toFixed(2)+" L/100 km";
  return(result);
}

function converter_internal_time(v)
{
  var result=new Object();
  result.inValue=v;
  // Removing the final "?" kills "7:00 pm"
  var re = /([0-9]{1,2})(:?([0-9]{2}))?(:[0-9]{2})?[\s]*(am|pm|AM|PM|a\.m\.|p\.m\.|A\.M\.|P\.M\.)?[\s]*([+\-0-9A-Z]{3,5})?/;
  var regexp_result = re.exec(v);
  if (regexp_result[3]==undefined) {
    regexp_result[3]="00";
  }
  if (regexp_result[4]==undefined) {
    regexp_result[4]='';
  }
  if (regexp_result[5]) { // 12h -> 24h
    if (regexp_result[1]=='12') {
      regexp_result[1]=0;
    }
    if (regexp_result[5].toLowerCase().match(/pm|p\.m\./)) {
      regexp_result[1]=String(Math.abs(regexp_result[1])+12);
    }
    var end_result=regexp_result[1]+':'+regexp_result[3]+regexp_result[4];
    result.inUnit='12h';
    result.outUnit='24h';
  } else { // 24h -> 12h
    if (regexp_result[1]=='00' || regexp_result[1]=='0') {
      regexp_result[1]=12;
    }
    if (regexp_result[1]>12) {
      end_result=String(Math.abs(regexp_result[1])-12)+':'+regexp_result[3]+regexp_result[4]+' PM';
    } else {
      end_result=regexp_result[1]+':'+regexp_result[3]+regexp_result[4]+' AM';
    }
    result.inUnit='24h';
    result.outUnit='12h';
  }
  result.outValue=end_result;
  result.not_numerical=true;
  result.only_show_converted=true;
  var local_tz=gConverterTZ.get_local_offset();
  var orig_tz=gConverterTZ.get_tz_offset(regexp_result[6]);
  if (isNaN(orig_tz) || orig_tz==local_tz) {
    // No timezone spec, or same timezone -- returning converted, if applicable
    if (
      (result.outUnit=='24h' && gConverterPrefs.getPref('pref_24h')) ||
      (result.outUnit=='12h' && !gConverterPrefs.getPref('pref_24h'))
    ) {
      return(result);
    } else {
      return({inhibit:true});
    }
  }
  var hh=regexp_result[1]; // regexp_result[1] is ALWAYS 24h, see above
  var mm=regexp_result[3];
  var ss=regexp_result[4];
  //var tz=regexp_result[5];
  var delta_tz=local_tz-orig_tz; // to be added to the orig time in order to get local time
  var delta_tz_hh=Math.floor(delta_tz);
  var delta_tz_mm=(delta_tz-delta_tz_hh)*60;
  var local_hh=1*hh;
  var local_mm=1*delta_tz_mm+1*mm;
  if (local_mm>59) {
    local_hh++;
    local_mm-=60;
  }
  if (local_mm<0) {
    local_hh--;
    local_mm+=60;
  }
  if (local_mm<10) {
    local_mm="0"+local_mm;
  }
  local_hh+=delta_tz_hh;
  var local_dd=0;
  if (local_hh>23) {
    local_dd++;
    local_hh-=24;
  }
  if (local_hh<0) {
    local_dd--;
    local_hh+=24;
  }
  if (local_dd==0) {
    local_dd='';
  } else if (local_dd==1) {
    local_dd='/+1';
  } else {
    local_dd='/-1';
  }
  result.inUnit='remote TZ';
  result.outUnit='local TZ';
  if (gConverterPrefs.getPref('pref_24h')) {
    result.outValue=local_hh+':'+local_mm+ss+local_dd;
    return(result);
  } else {
    if (local_hh==0) {
      local_hh=12;
    }
    if (local_hh>12) {
      local_hh-=12;
      var local_ampm=' PM';
    } else {
      var local_ampm=' AM';
    }
    result.outValue=local_hh+':'+local_mm+ss+local_dd+local_ampm;
    return(result);
  }
}

function converter_sqft2sqm(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v*0.09290304;
  result.inUnit="sq ft";
  result.outUnit="m"+String.fromCharCode(178);
  return(result);
}

function converter_sqm2sqft(v)
{
  var result=new Object();
  result.inValue=v;
  result.outValue=v/0.09290304;
  result.inUnit="m"+String.fromCharCode(178);
  result.outUnit="sq ft";
  return(result);
}

function converter_nm2ftlb(v)
{
  var result=new Object();
  //result.only_show_converted=true;
  result.not_numerical=true;
  result.inValue=v;
  var outFt=v*0.7375621;
  var outIn=v*0.7375621*12;
  outFt=Math.round(outFt*1000)/1000;
  outIn=Math.round(outIn*1000)/1000;
  result.inUnit="Nm";
  result.outCustom=outFt+" ft-lb; "+outIn+" in-lb";
  return(result);
}

/*

  ====================================================================
                              ADJUSTERS
  ====================================================================

  These functions perform order of magnitude adjustments within the same
  units system, trying to guess the most intuitive unit. For instance
  it would be counter-intuitive to show a distance of the order of
  hundreds of kilometers in millimeters.
  
*/

function metricWeightAdjustments(o)
{
  if (o.outUnit!="kg") {
    if (o.outUnit=='mg') {
      return(o);
    }
    return(o);
  }
  var tv=o.outValue;
  if ((tv>=1) && (tv<1000)) {
    return(o);
  }
  if (tv>=1000) {
    o.outValue/=1000;
    o.outUnit="t";
    return(o);
  }
  if (tv>=0.001) {
    o.outValue*=1000;
    o.outUnit="g";
    return(o);
  }
  o.outValue*=1000000;
  o.outUnit="mg";
  return(o);
}

// I'm not familiar enough with the imperial units, these may not be the most
// intuitive choices; I hope they're acceptable though -- if not, you're free
// to send me a patch any time! :)
function imperialDistanceAdjustments(o)
{
  if (o.outUnit!="in") {
    if (o.outUnit=="mil") {
      return(o);
    }
    return(o);
  }
  var tv=o.outValue;
  if (tv<(1.0/32.0)) {
    o.outValue=tv*100;
    o.outUnit="mil";
    return(o);
  }
  if (tv>=5280*36) {
    o.outValue=tv/12/52.8/100;
    o.outUnit="mi";
    return(o);
  }
  if (tv>=36000) {
    o.outCustom=converter_smartRound(tv/36)+" yd ("+converter_smartRound(tv/12/52.8/100)+" mi)";
    return(o);
  }
  if (tv>=36) {
    o.outCustom=converter_smartRound(tv/36)+" yd ("+converter_smartRound(tv/12)+" ft)";
    return(o);
  }
  if (gConverterPrefs.getPref('pref_in_fract')) {
    var tv32=Math.round(tv*32) % 32;
    if (tv32) {
      if (tv32==16) {
        var fraction="1/2";
      } else if (!tv32 % 8) {
        var fraction=(tv32/8)+"/4";
      } else if (!tv32 % 4) {
        var fraction=(tv32/4)+"/8";
      } else if (!tv32 % 2) {
        var fraction=(tv32/2)+"/16";
      } else {
        var fraction=tv32+"/32";
      }
    }
    tv=Math.floor(tv);
    var inches=tv % 12;
    tv=(tv-inches)/12;
    o.outCustom=tv+"'"+inches;
    if (fraction) {
      o.outCustom=o.outCustom+"-"+fraction+"\"";
    } else if (tv==0) {
      o.outCustom=inches+"\"";
    } else {
      o.outCustom=o.outCustom+"\"";
    }
    return(o);
  } else {
    o.outValue=tv;
    o.outUnit='in';
    return o;
  }
}

function metricDistanceAdjustments(o)
{
  if (o.outUnit!="m") {
    if (o.outUnit==String.fromCharCode(181)+"m") {
      return(o);
    }
    return(o);
  }
  var tv=o.outValue;
  if ((tv>=1) && (tv<1000)) {
    return(o);
  }
  if (tv>=1000) {
    o.outValue/=1000;
    o.outUnit="km";
    return(o);
  }
  if (tv>=0.01) {
    o.outValue*=100;
    o.outUnit="cm";
    return(o);
  }
  if (tv>=0.0001) {
    o.outValue*=1000;
    o.outUnit="mm";
    return(o);
  }
  o.outValue*=1000000;
  o.outUnit=String.fromCharCode(181)+"m";
  return(o);
}

function imperialVolumeAdjustments(o)
{
  if ((o.outUnit=='fl oz') && (o.outValue>80)) {
    o.outUnit='gal';
    o.outValue/=128;
  }
  if ((o.outUnit=='gal') && (o.outValue<0.625)) {
    o.outUnit='fl oz';
    o.outValue*=128;
  }
  return(o);
}

function metricVolumeAdjustments(o)
{
  if ((o.outUnit=='ml') && (o.outValue>1000)) {
    o.outUnit='L';
    o.outValue/=1000;
  }
  if ((o.outUnit=='L') && (o.outValue<1)) {
    o.outUnit='ml';
    o.outValue*=1000;
  }
  return(o);
}

/*

  ____________________________________________________________________

   Typically you shouldn't need to change anything below this line.
  ____________________________________________________________________



  ====================================================================
                           REGISTRATION
  ====================================================================
  
  This is the Mozilla internal stuff, where we register the converter
  entry in the popup. Traditionally this stays at the beginning of the
  script, but I preferred to leave the converters at the top so it's as
  easy as possible for future developers to write new converters.
  
  If you need to write new adjusters, you'll have to register them
  as an if case within converterPopup(). Use the existing adjusters
  as a model.

*/

window.addEventListener("load",converterInit,false);

var converterBundle;
var converter_extended_symbols=new Array();
var converter_extended_symbols_raw=new Array();
var converter_symbol_1=new String();
var converter_symbol_2plus=new String();
var conversions = registerConverters();

function converter_showLocation(loc)
{
	var newTab = gBrowser.addTab(loc);
	gBrowser.selectedTab = newTab;
}

function converterDoWelcome()
{
  var prefs = Components.classes["@mozilla.org/preferences-service;1"].
                      getService(Components.interfaces.nsIPrefService);
  var ver='0.7.1';
  var decision=false;
  prefs = prefs.getBranch("extensions.converter.");
  // IMPORTANT! The order of the tests below is significant!
  if (
    (prefs.getPrefType('current_version')==prefs.PREF_INVALID) ||
    (prefs.getCharPref('current_version')=='')
  ) {
    prefs.setCharPref('current_version',ver);
    decision='install';
  }
  if (
    (prefs.getPrefType('first_version')==prefs.PREF_INVALID) ||
    (prefs.getCharPref('first_version')=='')
  ) {
    prefs.setCharPref('first_version',prefs.getCharPref('current_version'));
  }
  if (prefs.getCharPref('current_version')!=ver) {
    prefs.setCharPref('current_version',ver);
    decision='upgrade';
  }
  if (!decision) {
    return(false);
  }
  var fver=prefs.getCharPref('first_version');
  if (decision=='install') {
    if (
      confirm(
        "Congratulations!\n"+
        "-------------------------------------\n"+
        "\n"+
        "You have successfully installed the Converter extension!\n"+
        "\n"+
        "Would you like to find out how to use it?"
      )
    ) {
      converter_showLocation("http://converter.mozdev.org/feedback/welcome.html?ver="+ver);
    }
  }
  if (decision=='upgrade') {
    if (
      confirm(
        "Congratulations!\n"+
        "-------------------------------------\n"+
        "\n"+
        "You have successfully upgraded the Converter extension!\n"+
        "\n"+
        "Would you like to find out what's new?"
      )
    ) {
      converter_showLocation("http://converter.mozdev.org/feedback/whatsnew.html?ver="+ver+"&fver="+fver);
    }
  }
  return(true);
}

function converterNewPage(objEvent)
{
  var cDoc=objEvent.originalTarget;
  var cBodyT=cDoc.getElementsByTagName("body");
  var cBody=(cBodyT.length > 0) ? cBodyT[0] : false;
  if (cBody) {
    cBody.setAttribute("converter_extension_converted",false);
  }
}

function converterOldPage(cDoc)
{
  var cBodyT=cDoc.getElementsByTagName("body");
  var cBody=(cBodyT.length > 0) ? cBodyT[0] : false;
  if (cBody) {
    cBody.setAttribute("converter_extension_converted",true);
  }
}

function converterIsNewPage(cDoc)
{
  var cBodyT=cDoc.getElementsByTagName("body");
  var cBody=(cBodyT.length > 0) ? cBodyT[0] : false;
  if (cBody) {
    var tmp=(cBody.getAttribute("converter_extension_converted")=="true")?true:false;
    return(!tmp);
  }
  // This should be meaningless, as far as I can tell
  return(true);
}

function converterInit()
{
  // Welcome first ------------------------------------------------------------
  converterDoWelcome();

  window.addEventListener('load',converterNewPage,true);

  // Now the proper stuff -----------------------------------------------------
  converterBundle=document.getElementById("converter_string_bundle");
  var tmpElement=document.getElementById("contentAreaContextMenu");
  if (tmpElement)
    tmpElement.addEventListener("popupshowing",onConverterPopup,false);
  converterInitControls();
}

function converterInitControls()
{
  document.getElementById('converter_status').hidden = !gConverterPrefs.getPref('pref_fullpage_icon');
}

/**
* This function is executed when the popup menu pops up. It just triggers
* converter_doConversion() and, depending on the result, shows the result
* in the popup menu.
*/
function onConverterPopup()
{
  var item = document.getElementById("context_converterselect");
  if (!gContextMenu.isTextSelected){
    if (!gConverterPrefs.getPref('pref_fullpage_menu')) {
      item.hidden=true;
      return true;
    }
    if (!converterIsNewPage(content.document)) {
      item.hidden = true;
      return(true);
    }
    item.hidden = false;
    item.setAttribute("label", 'Convert the entire page');
    return(true);
  }
  if (!gConverterPrefs.getPref('pref_selection_menu')) {
    item.hidden=true;
    return true;
  }
  if (gContextMenu.searchSelected) {
    // FF 1.x
    var sel=gContextMenu.searchSelected();
  } else {
    // FF 2.x
    var sel=getBrowserSelection();
  }
  var cr=converter_doConversion(sel,false);
  if (!cr) {
    item.hidden=true;
    return(true);
  }

  item.hidden = false;
  var label = cr.inValue + " " + cr.inUnit + " = " + cr.outCustom;
  item.setAttribute("label", label);
  return(true);
}

/**
* This function is called when the user clicks on the converter entry
* in the popup.
*/
function onConvert()
{
  if (!gContextMenu.isTextSelected)
    converter_convertPage();
  return(true);
}

/*

  ====================================================================
                         INTERNAL CONVERTER CODE
  ====================================================================
  
  These functions are internal to the converter itself, they perform
  various tasks documented above each function.
  
*/

function converter_regressionTest()
{
  if (content.document.title!="Converter Regression Test Page {8B72860F-C5F8-4286-865E-D2C2DB98A9E6}") {
    return false;
  }
  var tests=content.document.getElementsByName('converter_test');
  var c_assertions=content.document.getElementsByName('contextual_assertion');
  var c_test=content.document.getElementsByName('contextual_test');
  var p_assertions=content.document.getElementsByName('full_page_assertion');
  var p_test=content.document.getElementsByName('full_page_test');
  var i=0;
  var color_passed='#AAFFAA';
  var color_failed='#FFAAAA';
  var status_text='Some tests have failed:<br><ul>';
  var status_result=true;
  for(i=0;i<tests.length;i++) {
    var test=tests[i].innerHTML;

    // contextual conversion
    var cr=converter_doConversion(test,false);
    if (!cr) {
      if (c_assertions[i].innerHTML=='[no match]') {
        c_test[i].style.color='#000000';
        c_test[i].style.backgroundColor=color_passed;
        c_test[i].innerHTML='PASSED';
        c_test[i].style.fontWeight='bold';
        c_test[i].style.textAlign='center';
      } else {
        c_test[i].style.backgroundColor=color_failed;
        c_test[i].innerHTML='[no match]';
      }
    } else {
      var contextual=cr.inValue + " " + cr.inUnit + " = " + cr.outCustom;
      c_test[i].style.color='#000000';
      if (contextual==c_assertions[i].innerHTML) {
        c_test[i].style.backgroundColor=color_passed;
        c_test[i].innerHTML='PASSED';
        c_test[i].style.fontWeight='bold';
        c_test[i].style.textAlign='center';
      } else {
        c_test[i].style.backgroundColor=color_failed;
        c_test[i].innerHTML=contextual;
        status_result=false;
        status_text+="<li>Test "+(i+1)+" has failed (contextual).</li>";
      }
    }

    // full page conversion
    var full_page=converter_convertText(test);
    p_test[i].style.color='#000000';
    if (full_page==p_assertions[i].innerHTML) {
      p_test[i].style.backgroundColor=color_passed;
      p_test[i].innerHTML='PASSED';
      p_test[i].style.fontWeight='bold';
      p_test[i].style.textAlign='center';
    } else {
      p_test[i].style.backgroundColor=color_failed;
      p_test[i].innerHTML=full_page;
      status_result=false;
      status_text+="<li>Test "+(i+1)+" has failed (full page).</li>";
    }
  }

  var stat_p=content.document.getElementById('crt_status_p');
  var stat_span=content.document.getElementById('crt_status_span');
  if (status_result) {
    stat_p.style.backgroundColor=color_passed;
    stat_span.innerHTML='All tests passed successfully.';
  } else {
    stat_p.style.backgroundColor=color_failed;
    stat_span.innerHTML=status_text+"</ul>";
  }
  
  return true;
}

/*
  This function swiped from http://www.mredkj.com/javascript/nfbasic.html
*/
function converter_addCommas(nStr)
{
	nStr += '';
	var x = nStr.split('.');
	var x1 = x[0];
	var x2 = x.length > 1 ? '.' + x[1] : '';
	var rgx = /(\d+)(\d{3})/;
	while (rgx.test(x1)) {
		x1 = x1.replace(rgx, '$1' + ',' + '$2');
	}
	return x1 + x2;
}

function converter_smartRound(v)
{
  var vStr=new String(v);
  // special case: no decimals--return as such
  if (vStr.indexOf('.')==-1)
    return(converter_addCommas(v));
  // special case: two decimals at most--return as such
  /*
  // Not really useful, I have decided after all.
  // I might yet revive it though, so I just commented it out for now.
  if (vStr.indexOf('.')>vStr.length-4)
    return(converter_addCommas(v));
  */
  var done=false;
  var out=new Number(v);
  var absV=Math.abs(v);
  for(var val_e=2;!done && val_e>-17;val_e--) {
    var val=eval('Number(1e'+val_e+')');
    if (absV>val) {
      var out=out.toFixed(-val_e+2);
      done=true;
    }
  }
  out=new String(out);
  while(out.indexOf('.')>-1 && out.substr(-1,1).match(/[0.]/)) {
    out=new String(out.substr(0,out.length-1));
  }
  if (Number(out)>=1000) {
    out=converter_addCommas(out);
  }
  return(out);
}

/*
* This function is the conversion brain -- it takes in a text, checks for
* values to convert, converts them, executes the adjusters, and if all goes
* well, returns a converter result object, and false otherwise.
*/
function converter_doConversion(txt,sw)
{
  if (sw) {
    var temp = sw;
  } else {
    var temp = converterGetSelectedWord(txt);
    if (!temp) {
      return(false);
    }
  }
  var pos=inArray(temp[1],conversions[0]);
  if (pos==-1) {
    return(false);
  }
  if (temp[1]!='internal_time') {
    temp[0]=determine_separators(temp[0]);
  }
  if (temp[0]==false) {
    return(false);
  }

  var converter=conversions[1][pos];
  // teh problem ist hier ------------------\
  var cr=eval("new converter_"+converter+"(\""+temp[0]+"\")"); // conversion result
  if (temp[4]>1) {
    cr.inValue*=temp[4];
    cr.outValue*=temp[4];
  }
  // This is rather academic, since we're only using it once below, but hey!
  // It's a hobby project!
  if (cr.outUnit=="m") {
    cr=metricDistanceAdjustments(cr);
  }
  if (cr.outUnit=="kg") {
    cr=metricWeightAdjustments(cr);
  }
  if (cr.outUnit=="in") {
    cr=imperialDistanceAdjustments(cr);
  }
  if ((cr.outUnit=='gal') || (cr.outUnit=='fl oz')) {
    cr=imperialVolumeAdjustments(cr);
  }
  if ((cr.outUnit=='ml') || (cr.outUnit=='L')) {
    cr=metricVolumeAdjustments(cr);
  }
  if (!cr.not_numerical) {
    cr.inValue=converter_smartRound(cr.inValue);
    cr.outValue=converter_smartRound(cr.outValue);
  }
  if (!cr.outCustom) {
    cr.outCustom=cr.outValue+" "+cr.outUnit;
  }
  return(cr);
}

/**
* This function receives a string which presumably contains a number.
* The number may be formatted with either comma or period as decimal separator,
* and the other one as thousands separator. This function tries to determine
* which way it is, defaulting to English format if ambiguous.
* It returns a proper number.
*/
function determine_separators(s)
{
  // We need to get a string, otherwise we don't care at all
  if (!isNaN(s)) {
    return(s);
  }
    
  try {
    var tmp=s.indexOf(",");
  }
  catch(e) {
    return(s);
  }
    
  /*

  We used to execute the code below -- which worked well for legitimate
  numerical situations like "6.382.866 miles", but it was quite lousy
  at detecting sillier stuff such as <<disable that "version 6.2.2" crap>>,
  which it ended up converting as "622 inches".

  So this was replaced with the block of code just below the comment.

  // Ok, first we'll take the default out of the way
  if (tmp==-1) {
    s=s.replace(/\./g,"");
    return(s);
  }
  */

  // Even here, we need to check whether this makes any sense. Or not.
  if (tmp==-1) {
    // check whether we have a properly formatted number
    re = /^[\d]{1,3}(\.[\d]{3})*(\.[\d]+)$/
    if (re.exec(s)) {
      s=s.replace(/\./g,"");
      return(s);
    } else {
      return(false);
    }
  }

  // Now we're sure we have a comma in there somewhere. Let's dissect this:
  if (s.indexOf(".")!=-1) {
    // Good, we also have a period, this makes it easy to determine which case
    // it is: the last one is the decimal separator.
    if (s.lastIndexOf(".")>s.lastIndexOf(",")) {
      // If it's the period, we just remove the commas and return the string
      s=s.replace(/,/g,"");
    } else {
      // If it's the comma, we remove the periods, change the comma into a
      // period and return the string.
      s=s.replace(/\./g,"");
      s=s.replace(/,/g,".");
    }
  } else {
    // Ewww, this is nasty, we positively have a comma, but we have no
    // period to serve as a witness. We'll branch in two once again:
    if (s.indexOf(",")!=s.lastIndexOf(",")) {
      // Either there's more than one comma, in which case the period would
      // be the decimal separator...
      s=s.replace(/,/g,"");
    } else {
      // ...or there's a single comma. If there's only one comma, then the
      // number has only ony chance to be suspected of using comma as a
      // thousands separator: it MUST be in the proper position for it to be a
      // thousands separator AND the number must be no larger than 100,000.
      // In which case we'll ASSUME the default case (period is assumed to be
      // the decimal separator). This is not perfect, but it's the safest bet.
      // If either of the conditions above is not met, we'll use the comma
      // as decimal separator.
      if ((s.length<=7) && (s.indexOf(",")==s.length-4)) {
        s=s.replace(/,/g,"");
      } else {
        s=s.replace(/\./g,"");
        s=s.replace(/,/g,".");
      }
    }
  }
  return(s);
}

/*
* This function registers the converters defined at the beginning of this
* file. What it basically does is split that string into items and builds
* two arrays, one of which contains the units and the other the converters.
* It then builds an array with two elements which are the arrays previously
* described, and returns it. It is used by onConverterPopup().
*/
function registerConverters()
{
  var conv_symbols=new Array();
  var conv_functions=new Array();
  
  var atoms=conv_table.split(" ");
  var conv_symbol=true;
  var register_extended=true;
  if (converter_extended_symbols.length) {
    register_extended=false;
  }
  
  var i=0;
  var a_converter_symbol_1=new Array();
  var a_converter_symbol_2plus=new Array();
  var dbg_i=0;
  
  for(i=0;i<atoms.length;i++) {
    var atom=atoms[i];
    if (!atom) {
      continue;
    }
    if (conv_symbol) {
      if (register_extended && (atom.indexOf('_')>-1)) {
        converter_extended_symbols_raw[converter_extended_symbols_raw.length]=atom.replace(/_/g,' ');
        converter_extended_symbols[converter_extended_symbols.length]=atom.replace(/_/g,'[\\s]+');
      }
      conv_symbols[conv_symbols.length]=atom;
      if (i<dbg_i) {
        alert('atom: '+atom+' ('+converter_toASCII(atom)+')');
      }
      var letter=atom.substring(0,1);
      if (i<dbg_i) {
        alert("Atom: "+atom+"; letter: "+letter);
      }
      if (inArray(letter,a_converter_symbol_1)==-1) {
        a_converter_symbol_1[a_converter_symbol_1.length]=letter;
//        if (converter_symbol_1==undefined) {
//          converter_symbol_1=letter;
//        } else {
          converter_symbol_1+=letter;
//        }
      }
      var j=0;
      for(j=1;j<atom.length;j++) {
        letter=atom.substring(j,j+1);
        if (inArray(letter,a_converter_symbol_2plus)==-1) {
          a_converter_symbol_2plus[a_converter_symbol_2plus.length]=letter;
//          if (converter_symbol_2plus==undefined) {
//            converter_symbol_2plus=letter;
//          } else {
            converter_symbol_2plus+=letter;
//          }
        }
      }
      if (i<dbg_i) {
        alert(
          "first: "+a_converter_symbol_1.toString()+"\n"+
          "2plus: "+a_converter_symbol_2plus.toString()
        );
      }
    } else {
      conv_functions[conv_functions.length]=atom;
    }
    conv_symbol=!conv_symbol;
  }
  converter_symbol_1=converter_regexpise(converter_symbol_1);
  converter_symbol_2plus=converter_regexpise(converter_symbol_2plus);
  
  var result=new Array(conv_symbols,conv_functions);
  return(result);
}

/**
* Transforms a string containing raw, individual characters to be matched
* by regexp to a form which is actually understood by match()
*/
function converter_regexpise(in_s)
{
//  alert('in_s: '+in_s);
  var i=0;
  var result=new String;
  var s=new String(in_s);
  var re_raw=/[0-9a-zA-Z !"#$%&'*+,.:;<=>?@_`{|}~]/;
  var re_escape=/[\(\)\-\/\[\\\]\^]/;
  for(i=0;i<s.length;i++) {
    var letter=s.substring(i,i+1);
    var replace=letter;
    // don't forget about the underscore -- it's never matched here!
    // standard stuff, goes as is
    if (!re_raw.exec(letter)) {
      // stuff that simply needs to be escaped with a backslash
      if (re_escape.exec(letter)) {
        replace='\\'+replace;
      } else {
        var hex=replace.charCodeAt(0).toString(16).toUpperCase();
        while(hex.length<4) {
          hex='0'+hex;
        }
        replace='\\u'+hex;
      }
    }
    result+=replace;
  }
//  alert('result: '+result);
  return result;
}

/**
* A simple function to emulate indexOf() for arrays. Returns the position
* of the needle in the haystack array, or -1 if not found. Used by
* onConverterPopup() to determine the units in the selection by matching
* the 2nd result of converterGetSelectedWord() below against the second array
* returned by registerConverters() above.
*/
function inArray(needle, haystack)
{
  var i=0;
  for(i=0;i<haystack.length;i++) {
    var hayAtom=haystack[i];
    if(needle==hayAtom) {
      return(i);
    }
  }
  return(-1);
}

/*
* This function converts the entire page. It basically breaks down like this:
* first it calls converter_identifyElement(), a recursive function which
* walks through all elements in the BODY, and when it finds a text element
* it converts it using converter_convertText(); converter_convertText() calls
* converterGetSelectedWord() on the entire text; if it finds a match, it converts
* it and executed converterGetSelectedWord() on the rest of the text -- and so
* on, until converterGetSelectedWord() doesn't find any more matches in the
* remainder of the string.
*/
function converter_convertPage()
{
  window.setCursor('wait');
  if (converter_regressionTest()) {
    window.setCursor("auto");
    return(true);
  }
  var myDocument=content.document;
  var myBrowser=gBrowser;
  if (myBrowser.contentWindow.frames.length>0) {
    for(var i=0;i<gBrowser.contentWindow.frames.length;i++) {
      var myDocument=gBrowser.contentWindow.frames[i].document;
      _converter_convertDoc(myDocument);
    }
  }
  _converter_convertDoc(gBrowser.contentWindow.document);
  window.setCursor("auto");
  return true;
}

function _converter_convertDoc(aDoc)
{
  converterOldPage(aDoc);
  if (aDoc.getElementsByTagName('BODY').length) {
    converter_identifyElement(aDoc.getElementsByTagName('BODY')[0]);
    return true;
  } else {
    return false;
  }
}

// see the documentation above on converter_convertPage() for details on this function
function converter_identifyElement(docu)
{
  var d=docu;
  if (d==undefined)
    return(false);
  do {
    if (d.nodeType==1) {
      if (d.nodeName!='SCRIPT')
        converter_identifyElement(d.firstChild);
    } else if (d.nodeType!=8) { // not converting HTML comments
      var d_content=d.nodeValue;
      var skip_sibling=false;
      if (d.nextSibling!=undefined && d.nextSibling.nodeName=='SUP') {
        var sup_index=d_content.length;
        d_content=d_content+'^'+d.nextSibling.firstChild.nodeValue;
        skip_sibling=true;
      }
      var re=/[0-9]/;
      var result=re.exec(d.nodeValue);
      if (!result) {
        d=d.nextSibling;
        if (skip_sibling) {
          skip_sibling=false;
          if (d!=undefined)
            d=d.nextSibling;
        }
        continue;
      }
      //alert('Converting this: '+d.nodeValue);
      var new_content=converter_convertText(d_content);
      //d.nodeValue=converter_convertText(d_content);
      if (new_content!=d_content) {
        var new_index=sup_index+new_content.length-d_content.length;
        if (new_content.substring(new_index)!=d_content.substring(sup_index)) {
          d.nodeValue=new_content;
          if (skip_sibling) {
            d.parentNode.removeChild(d.nextSibling);
          }
        } else {
          d.nodeValue=new_content.substring(0,new_index); // no "+1" because we clear the artificial "^" as well
        }
      }
    };
    d=d.nextSibling;
    if (skip_sibling) {
      skip_sibling=false;
      if (d!=undefined)
        d=d.nextSibling;
    }
    // debug: alert('d.nodeType='+d.nodeType+'; d.nodeName='+d.nodeName+'; d.nodeValue='+d.nodeValue);
  } while (d!=undefined);
  return(true);
}

/*
* Just a debug function which is not usually needed
*/
function converter_toASCII(str)
{
  var dumpData=new String();
  for(var i=0;i<str.length;i++) {
    dumpData=dumpData+str.charCodeAt(i)+', ';
  }
  return(dumpData);
}

// see the documentation above converter_convertPage() for details on this function
function converter_convertText(txt)
{
  var rawText=new String(txt); // this is the original text which gets chopped
  //debug: alert('text before: |'+rawText+'|');
  var re = /\r/g;
  rawText=rawText.replace(re,' ');
  re = /\n/g;
  rawText=rawText.replace(re,' ');
  //debug: alert('text after: |'+rawText+'|');
  var convertedText=''; // this is the converted text which grows
  var wordData=false; // temporary var where we store the result of cGSW()

  wordData=converterGetSelectedWord(rawText)
  while (wordData) {
    convertedText=convertedText+rawText.substring(0,wordData[3]+wordData[2].length);
    rawText=rawText.substring(wordData[3]+wordData[2].length,rawText.length);
    var cr=converter_doConversion(wordData[2],wordData);
    if (cr) {
      if (!cr.inhibit) {
        if (cr.only_show_converted) {
          convertedText+=' ('+cr.outCustom+')';
        } else {
          convertedText=convertedText+' ('+cr.inValue + " " + cr.inUnit + " = " + cr.outCustom+')';
        }
      }
    }
    wordData=converterGetSelectedWord(rawText)
  }
  convertedText=convertedText+rawText;
  return(convertedText);
}

/**
* This function parses the current selection, searching for the first
* substring which looks like a dimension. The typical dimension understood
* by this function is a substring made of a number, optionally followed by
* one or more spaces, followed by a word (loose definition here, check
* the source for the actual regexp). Before looking for the typical dimension
* however, it must look for the awkward imperial formats in
* feet/inches/fractions because otherwise it could end up only matching the
* feet in such a format.
*
* It returns an array made of two elements: the first one is the number and
* the second one is the unit.
*
* Please note that for typical measurements it also checks for fractions,
* and if found it tries to determine the decimal separator for each term and
* evaluates them numerically. The first element of the resulting array is
* therefore not guaranteed to be a string--in this case, it's a number.
*/
function converterGetSelectedWord(word)
{
  var Aresult1=new Array(0,0,0,0,0);
  var Aresult2=new Array(0,0,0,0,0);
  var Aresult3=new Array(0,0,0,0,0);
  var Aresult4=new Array(0,0,0,0,0);
  var Aresult5=new Array(0,0,0,0,0);
  var Aresult6=new Array(0,0,0,0,0);
  var Aresult7=new Array(0,0,0,0,0);
  var Aresult8=new Array(0,0,0,0,0);
  var Aresult9=new Array(0,0,0,0,0);
  var Aresult10=new Array(0,0,0,0,0);

  // Let's start by checking for the awkward Imperial notations.
  
  // First checking for egineering format (1'2 1/32" or 1'2-1/32").
  // If found, we convert the whole thing to inches and return as if we
  // encountered the corresponding string in inches using decimal notation
  // instead of fractions.
  var re = /(([0-9]*)\')?([0-9]+)[\s\-]([0-9\/]+)\"/;
  var result = re.exec(word);
  if (result) {
    if (!result[1])
      result[1]=0;
    Aresult1[0]=new Number(0);
    if (result[2]!=undefined) {
      Aresult1[0]+=new Number(12*result[2]);
    }
    if (result[3]!=undefined) {
      Aresult1[0]+=new Number(result[3]);
    }
    if (result[4]!=undefined) {
      Aresult1[0]+=new Number(eval(result[4]));
    }
    Aresult1[1]="in";
    Aresult1[2]=new String(result[0]);
    Aresult1[3]=word.indexOf(result[0]);
  }
  
  var re = /([0-9]*)\'([0-9]+)[\s\-]([0-9\/]+)/;
  var result = re.exec(word);
  if (result) {
    Aresult7[0]=12*result[1]+1*result[2]+eval(result[3]);
    Aresult7[1]="in";
    Aresult7[2]=new String(result[0]);
    Aresult7[3]=word.indexOf(result[0]);
  }
  
  // And finally checking for informal format (5'10)
  // We first try finding the full format (x'y"), and if that doesn't work,
  // we look for the simplified format (x'y) -- unfortunately we can't append
  // ["]? at the end of the first regexp, because it would never match the
  // final double quote, even if it's there, since it's optional and there's
  // nothing afterwards to force the regexp algorithm to use it.
  var re = /([0-9]+)\'[ ]*([0-9]{1,2})\"/;
  var result = re.exec(word);
  if (result) {
    Aresult2[0]=12*result[1]+1*result[2];
    Aresult2[1]="in";
    Aresult2[2]=new String(result[0]);
    Aresult2[3]=word.indexOf(result[0]);
  } else {
    var re = /([0-9]+)\'[ ]*([0-9]{1,2})/;
    var result = re.exec(word);
    if (result) {
      Aresult2[0]=12*result[1]+1*result[2];
      Aresult2[1]="in";
      Aresult2[2]=new String(result[0]);
      Aresult2[3]=word.indexOf(result[0]);
    }
  }

  // Let's check for extended symbols
  // Double backslashes below because we need to escape the backslash itself to
  // get it across the string format. One hour of cursing; idiot, I should have known this.
  // Changed
  // ([\\-]?[0-9.][0-9.,\\/]*)[\\s]+
  // to
  // ([\\-]?[0-9.][0-9.,\\/]*)[\\s]*
  // in order to accept "10&deg; F"
  var re = new RegExp("([\\-]?[0-9.][0-9.,\\/]*)[\\s]*("+converter_extended_symbols.join('|').replace(/\./g,"\\.")+")","i");
  var result = re.exec(word);
  if ((result) && (result[1]) && (result[2])) {
    var pos=inArray(result[2].toLowerCase().replace(/[\s]+/g,' '),converter_extended_symbols_raw);
    if (pos>-1)
    {
      //debug: alert('extended match: '+result[1]+'--'+result[2]);
      Aresult5[0]=result[1];
      Aresult5[1]=converter_extended_symbols_raw[pos].replace(/ /g,'_');
      Aresult5[2]=new String(result[0]);
      Aresult5[3]=word.indexOf(result[0]);
/*
    } else {
      alert('extended match, no pos: '+result[1]+'--'+result[2]+'|'+re.source);
      alert('here: '+converter_extended_symbols.join('|'));
*/
    }
  }
  
  // 12h/24h
  //var re = /([0-9]{1,2})(:([0-9]{2}))?(:([0-9]{2}))?[\s]*(am|a\.m\.|pm|p\.m\.)?[\s]*([+\-0-9A-Z]{3,5})?/i;
  /*
    This needs to be broken down in two cases ("hh:mm" plus optional stuff, and
    "hh AM/PM" plus optional stuff); making both ":mm" and "AM/PM" optional
    results in messy matches which break the processing.
    ----------
    Update: the first case above needs to be broken down in two cases as well:
    "<hh:mm> [am/pm] <TZ>" and "<hh:mm> [am|pm]" in order to properly parse
    "abc 9:00 AM cba -- abc 9:00 AM cbaza" -- and in the first branch, the <TZ>
    needs to actually be verified.
    ----------
    Update: in the TZ area, the [\s]* is included explicitly because it needs
    to live in the TZ atom; otherwise "local 9:00 AM in Austria" is improperly
    spaced.
    ----------
    Update: if/else doesn't work well when multiple time formats are shown
    one after another -- "10 a.m. * 10 p.m. * 10 A.M. * 10 P.M. * 13:33:33 UTC"
    ----------
    Update: the military format (0800 GMT) is used more frequently than I
    had assumed; made the colon optional in all formats in order to support it.
    ----------
    Update: the conditions for Aresult10 used to be as follows:
    if (result && result[1] && (!result[3] || (result[4] && gConverterTZ.get_tz_offset(result[4])!=undefined) || (result[5] && gConverterTZ.get_tz_offset(result[5])!=undefined))) {

    I have removed the "!result[3]" condition because it didn't seem to make
    sense -- result[3] is a blanket generalization and I can't currently
    understand why it was there in the first place...
  */
  var re = /([0-9]{1,2}):?([0-9]{2})(:([0-9]{2}))?[\s]*(am|a\.m\.|pm|p\.m\.)?([\s]*([+\-][0-9]{4})|[\s]*([a-z]{3,4}))?/i;
  var result = re.exec(word);
  if (result && result[1] && result[2] && (!result[6] || (result[7] && gConverterTZ.get_tz_offset(result[7])!=undefined) || (result[8] && gConverterTZ.get_tz_offset(result[8])!=undefined))) {
    Aresult6[0]=result[0];
    Aresult6[1]='internal_time'
    Aresult6[2]=result[0];
    Aresult6[3]=word.indexOf(result[0]);
  }

  var re = /([0-9]{1,2}):?([0-9]{2})(:([0-9]{2}))?[\s]*(am|a\.m\.|pm|p\.m\.)?/i;
  var result = re.exec(word);
  if (result && result[1] && result[2]) {
    Aresult9[0]=result[0];
    Aresult9[1]='internal_time'
    Aresult9[2]=result[0];
    Aresult9[3]=word.indexOf(result[0]);
  }

  var re = /([0-9]{1,2})[\s]*(am|a\.m\.|pm|p\.m\.)([\s]*([+\-][0-9]{4})|[\s]*([A-Z]{3,4}))?/i;
  var result = re.exec(word);
  if (result && result[1] && ((result[4] && gConverterTZ.get_tz_offset(result[4])!=undefined) || (result[5] && gConverterTZ.get_tz_offset(result[5])!=undefined))) {
    if (!result[3]) {
      result[3]='';
    } else {
      result[3]=' '+result[3];
    }
    Aresult10[0]=result[1]+':00 '+result[2]+result[3];
    Aresult10[1]='internal_time'
    Aresult10[2]=result[0];
    Aresult10[3]=word.indexOf(result[0]);
  }

  // Ok, didn't find any of the wicked ones, let's check for
  // some reasonable measurements AND fractions.
  // --- NOTES ---
  // The last bracket used to include "\-", but I removed it as to allow
  // converting all of those Imperial idiots' "50-foot monster" and so on.
  // Hope I didn't mess up, I can't remember whether there was a real reason
  // for that in here, or whether it was simply a precaution.
  // -------------
  // Also, I replaced the second atom,
  // ([^ .,!?;+\(\)0-9]+)
  // with
  // ([^ .,!?;+\(\)0-9][^ .,!?;+\(\)0-9'"]*)
  // as to allow stuff in quotes -- e.g. "25 kg" -- without messing up the
  // support for inches (") and feet (').
  // -------------
  // Disallowing numbers in the first position for units, because it was
  // breaking dates
  // ("Thu, 21 Dec 2000 16:59 -0030" --> it was catching "2000 1")
  // -------------
  // Added a "rough" regexp test before the actual test in order to rule out
  // long strings made of numbers, which made the regexp freeze for way
  // too long.
  // -------------
  // Changed exclusive tests ([^foo]+) with inclusive tests ([bar]+) in order
  // to avoid various types of mishaps (I had to constantly adjust the
  // restrictions, and I always failed making it work for all situations).
  // The CVS version which still contains (imperfect) exclusive tests is 1.42.
  // -------------
  // The reason why I allow ([\\d]+[.]?)+ instead of ([\\d])+[.]? is that
  // I have to account for "6.322.867 feet". Of course, that shouldn't
  // interfere with 'disabling "Converter 0.6.2" stops FF from freezing',
  // which used to happen (it used to converte "062 in" instead of "6.2 in").
  // - END NOTES -

  // Not using the "proper" stuff here, we just need a rough test (never tested
  // this version, but it's most probably too strict to be useful, because we
  // include restrictions based on units, but be DON'T allow stuff
  // related to other considerations ("thousands", number formatting, etc)
  //var re = new RegExp("([\\-]?)([\\d,.]+)[\\s]*(["+converter_symbol_1+"]["+converter_symbol_2plus+"]+)");
  // ok, now that we have proper, identical rules for comma- and
  // period-fractions, we need better safeguards too.
  // var re = /([\-]?)([\d,.]+)[ ]*([^ .,!?;+\(\)0-9\s]+)/[ignore this bracket]

  var re = /([\-]?)([\d]+\.|[\d]+|\.|[\d]+,)([\s]*|-)([\d]*)([^ .,!?;+\(\)0-9\s]+)/
  if (re.exec(word)) {
    var re = new RegExp("([\\-]?)(((([\\d]+[.]?)+(,[0-9]+)?)|([\\d]*\\.[\\d]+))([/][0-9]+)?)[\\s]*(thousand|million)?[\\s]*(["+converter_symbol_1+"]["+converter_symbol_2plus+"]*)","i");
    var result = re.exec(word);
    //debug: alert('Matching this: '+word);
    // Must have the first (result), or the whole thing blows for null results
    if ((result) && (result[2]) && (result[10])) {
      //debug: alert('Match -- ['+result[1]+'] -- ['+result[2]+']');
      var tmpTest=new String(result[2]);
      var tmp_idx=result[2].indexOf("/");
      if (tmp_idx!=-1 && tmp_idx!=result[2].length-1) {
        var resultMembers=result[2].split("/");
        for(i=0;i<resultMembers.length;i++) {
          resultMembers[i]=determine_separators(resultMembers[i]);
        }
        result[2]=eval(resultMembers.join("/"));
      }
      var tmp=1;
      if (result[9]) {
        if (result[9]=='million') {
          tmp=1000000;
        } else {
          tmp=1000;
        }
      }
      Aresult3[0]=result[1]+result[2];
      Aresult3[1]=converter_simplify(result[10].toLowerCase());
      Aresult3[2]=converter_simplify(new String(result[0]));
      Aresult3[3]=word.indexOf(result[0]);
      Aresult3[4]=tmp;
    }

    var re = new RegExp("([\\-]?)((([\\d]+[\\,]?)+(\\.[0-9]+)?)([/][0-9]+)?)[\\s]*(thousand|million)?[\\s]*(["+converter_symbol_1+"]["+converter_symbol_2plus+"]*)","i");
    var result = re.exec(word);
    //debug: alert('Matching this: '+word);
    // Must have the first (result), or the whole thing blows for null results
    if ((result) && (result[2]) && (result[8])) {
      //debug: alert('Match -- ['+result[1]+'] -- ['+result[2]+']');
      var tmpTest=new String(result[2]);
      var tmp_idx=result[2].indexOf("/");
      if (tmp_idx!=-1 && tmp_idx!=result[2].length-1) {
        var resultMembers=result[2].split("/");
        for(i=0;i<resultMembers.length;i++) {
          resultMembers[i]=determine_separators(resultMembers[i]);
        }
        result[2]=eval(resultMembers.join("/"));
      }
      var tmp=1;
      if (result[7]) {
        if (result[7]=='million') {
          tmp=1000000;
        } else {
          tmp=1000;
        }
      }
      Aresult8[0]=result[1]+result[2];
      Aresult8[1]=converter_simplify(result[8].toLowerCase());
      Aresult8[2]=converter_simplify(new String(result[0]));
      Aresult8[3]=word.indexOf(result[0]);
      Aresult8[4]=tmp;
    }
  }

  // Explicitly looking for 10 foot 5, 10-foot-6, etc
  var re = /([0-9]+)[\s\-]*(foot|feet)[\s\-,]*([0-9]+)[\s\-]*(([0-9]+)\/([0-9]+)[\s\-]*)?(inches|inch)?/;
  var result = re.exec(word);
  if (result) {
    var tmp=0;
    if (result[4] && (result[6]>0)) {
      tmp=result[5]/result[6];
    }
    Aresult4[0]=12*result[1]+1*result[3]+tmp;
    Aresult4[1]='in';
    Aresult4[2]=new String(result[0]);
    Aresult4[3]=word.indexOf(result[0]);
  }
  
  var br=converter_bestResult(
    converter_bestResult(
      converter_bestResult(
        converter_bestResult(Aresult5,Aresult6),
        converter_bestResult(Aresult3,Aresult4)
      ),
      converter_bestResult(
        converter_bestResult(Aresult1,Aresult2),
        converter_bestResult(Aresult9,Aresult10)
      )
    ),
    converter_bestResult(Aresult7,Aresult8)
  );
  if (!br[2]) {
    return(false);
  }
  return(br);
}

function converter_simplify(word)
{
  while(word.substr(word.length-1)=='.') {
    word=word.substr(0,word.length-1);
  }
  return(word);
}

/*
* This function compares two converterGetSelectedWord() results and returns the
* best fit. By "best fit" I understand applying the following criteria, in this
* order (i.e. if all things are equal, go on; otherwise, make a decision):
* 1. The other is no match or it's an all-numeric match (i.e. the matching
*    string, at index 2 in the array, is empty or only made of 0-9 and space);
* 2. The other shows up later in the original string;
* 3. The other is shorter.
* 4. The other's VALUE is smaller -- this is here in order to prevent
*    Aresult3 (generic) from hijaking Aresult2 (e.g. "3'3"). Given that
*    this is practically the last meaningful test -- all hh:mm and so on
*    should have already been sorted out by now, and returing the first one
*    randomly (as the next decision would) can't be worse than comparing
*    their magnitudes.
* 5. The second is equally long to the first (in this case, return the first)
*/
function converter_bestResult(res1,res2)
{
  var re=/^[0-9 ]+$/;
  // (1)
  if (!res1[2] || re.exec(res1[2])) {
    return(res2);
  }
  if (!res2[2] || re.exec(res2[2])) {
    return(res1);
  }
  // (2)
  if (res1[3]<res2[3]) {
    return(res1);
  }
  if (res1[3]>res2[3]) {
    return(res2);
  }
  // (3)
  if (res2[2].length>res1[2].length) {
    return(res2);
  }
  if (res2[2].length<res1[2].length) {
    return(res1);
  }
  // (4)
  if (res2[0]>res1[0]) {
    return(res2);
  }
  // (5)
  return(res1);
}

function converter_statusClicked(evt)
{
  /*
  // Unfortunately icon switching doesn't work well with tabs, to my knowledge,
  // so we'll have to keep a single icon throughout. See linkification.
  var cvStatus = document.getElementById("converter-status-hbox");
  cvStatus.src = 'chrome://converter/skin/cv_stat_on.png';
  alert("Clicked!");
  */
  if (converterIsNewPage(content.document)) {
    converter_convertPage();
  }
}
